Attribute VB_Name = "modComplex"
'=======================================================================================
' Descrizione.....: Routines per le operazioni su
'                   numeri complessi.
' Nome dei Files..: modComplex.bas
' Data............: 15/02/2000
' Revisione.......: 10/01/2009 (Ver. 1.0.r).
' Revisione.......: 24/04/2009 (Ver. 1.0.r).
' Versione........: 1.0.r a 32 bits.
' Sistema.........: VB6 (SP5) sotto Windows XP (SP2).
' Scritto da......: F. Languasco 
' E-Mail..........: MC7061@mclink.it
' DownLoads a.....: http://www.flanguasco.org
'=======================================================================================
'   Comprende le seguenti funzioni e subroutines:
'
'   w = CCmp(r, i):     costruisce un numero complesso con
'                       parti reale ed immaginaria date.
'
'   w = CCon(z):        ritorna il coniugato di z.
'   w = CSom(z1, z2):   ritorna la somma z1 + z2.
'   w = CDif(z1, z2):   ritorna la differenza z1 - z2.
'   w = CMol(z1, z2):   ritorna il prodotto z1 * z2.
'   w = CDiv(z1, z2):   ritorna il rapporto z1 / z2.
'
'   w = CPtN(z1, N):    ritorna la potenza z1 ^ N.
'   w = CPtc(z1, z2):   ritorna la potenza z1 ^ z2.
'   w = CSqr(z):        ritorna la radice quadrata di z.
'   CNnr z, N, w():     calcola in w() le N radici Nesime di z.
'   w = CExp(z):        ritorna l' esponenziale di z.
'   w = CLog(z):        ritorna il logaritmo naturale di z.
'
'   w = CSin(z):        ritorna il seno di z.
'   w = CCos(z):        ritorna il coseno di z.
'   w = CTan(z):        ritorna la tangente di z.
'   w = CAsin(z):       ritorna l' arcoseno di z.
'   w = CAcos(z):       ritorna l' arcocoseno di z.
'   w = CAtn(z):        ritorna l' arcotangente di z.
'   w = CSinh(z):       ritorna il seno iperbolico di z.
'   w = CCosh(z):       ritorna il coseno iperbolico di z.
'   w = CTanh(z):       ritorna la tangente iperbolica di z.
'   w = CAsinh(z):      ritorna l' arcoseno iperbolico di z.
'   w = CAcosh(z):      ritorna l' arcocoseno iperbolico di z.
'   w = CAtnh(z):       ritorna l' arcotangente iperbolica di z.
'
'   r = CAbs(z):        ritorna il valore assoluto di z.
'   w = CModQ(z):       ritorna il modulo di z al quadrato in formato complesso.
'   r = CArg(z):        ritorna l' argomento di z.
'   r = ReaC(z):        ritorna la parte reale di z.
'   i = ImmC(z):        ritorna la parte immaginaria di z.
'   B = CEgu(z1, z2):   verifica l' eguaglianza z1 = z2.
'   CSWAP z1, z2:       scambia fra di loro i valori di z1 e z2.
'
'   w() = CPMPY(z1(), z2()): ritorna il prodotto dei polinomi z1() * z2().
'
'   S = CFormat$(z, F): ritorna una stringa rappresentante z con il formato F.
'   CMsgBox z:          espone un MsgBox con z a piena precisione.
'
'   r = DATAN2(y, x):   ritorna il valore dell' ArcoTangente
'                       di y/x su 4 Quadranti (-PI < DATAN2 <= PI).
'
'   Tipi:
'    i as Double, r as Double, x as Double, y as Double
'    w as Complex, z as Complex, z1 as Complex, z2 as Complex
'    N as Long
'    S as String, F as String
'    B as Boolean
'
'   Bibliografia:
'    Complex Variables
'    Murray R. Spiegel
'    Schaum's Outline Series
'    Pg. 2, 4, 35, 36.
'
'    Numerical Recipes in Fortran 77.
'    The Art of Scientific Computing.
'    Second Edition.
'    Volume 1 of Fortran Numerical Recipes.
'    William H. Press, Saul A. Teukolsky,
'    William T. Vetterling, Brian P. Flannery
'    Pg. 171, 172.
'
Option Explicit
'
Public Type Complex
    Re As Double  ' parte reale.
    Im As Double  ' parte immaginaria.
End Type
'
Private Const PI# = 3.14159265358979    ' 4# * Atn(1#)
Private Const PI_2# = PI / 2#           ' 90 in [rad].
Private Const SMALNO# = 1E-99
Public Function CPMPY(Cx() As Complex, Cy() As Complex) As Complex()
'
'   PURPOSE
'   MULTIPLY TWO POLYNOMIALS with complex coefficients
'
'   DESCRIPTION OF PARAMETERS
'   Z - VECTOR OF RESULTANT COEFFICIENTS, ORDERED FROM
'       SMALLEST TO LARGEST POWER       (Viene ritornato dalla funzione).
'   IDIMZ - DIMENSION OF Z (CALCULATED)
'   X - VECTOR OF COEFFICIENTS FOR FIRST POLYNOMIAL, ORDERED
'       FROM SMALLEST TO LARGEST POWER
'   IDIMX - DIMENSION OF X (DEGREE IS IDIMX-1)
'   Y - VECTOR OF COEFFICIENTS FOR SECOND POLYNOMIAL,
'       ORDERED FROM SMALLEST TO LARGEST POWER
'   IDIMY - DIMENSION OF Y (DEGREE IS IDIMY-1)
'
'   REMARKS
'   NONE
'
'   SUBROUTINES AND FUNCTION SUBPROGRAMS REQUIRED
'   NONE
'
'   METHOD
'   DIMENSION OF Z IS CALCULATED AS IDIMX + IDIMY - 1
'   THE COEFFICIENTS OF CZ ARE CALCULATED AS SUM OF PRODUCTS
'   OF COEFFICIENTS OF CX AND CY , WHOSE EXPONENTS ADD UP TO THE
'   CORRESPONDING EXPONENT OF CZ.
'
'   Liberamente tradotto (ed adattato alle mie esigenze) da:
'       PDP-10 Freeware Archives: 10-101 SSP, Version: 3, July 1973
'       Author: Sandia Laboratories (I.B.M.)
'       Revised by: H. David Todd, Wesleyan Univ., Middletown, CT
'       Source Language: FORTRAN IV     (Vedi file SSP.txt)
'
    Dim I&, J&, K&, IDIMX&, IDIMY&
'
    On Error GoTo CPMPY_ERR
'
    IDIMX = UBound(Cx)
    IDIMY = UBound(Cy)
    If (IDIMX * IDIMY < 1) Then Err.Raise 5, "CPMPY", _
                                          "Il grado di almeno un polinomio e' < 1."
'
    ReDim CZ(1 To IDIMX + IDIMY - 1) As Complex
'
    For I = 1 To IDIMX
        For J = 1 To IDIMY
            K = I + J - 1
            CZ(K) = CSom(CMol(Cx(I), Cy(J)), CZ(K))
        Next J
    Next I
'
    CPMPY = CZ()
'
'
CPMPY_ERR:
    If (Err.Number <> 0) Then
        Dim M$
        M$ = "Errore " & Str$(Err.Number) & vbNewLine
        M$ = M$ & Err.Description
        MsgBox M$, vbCritical, Err.Source
        Err.Clear
    End If
'
'
'
End Function
Public Function CModQ(z As Complex) As Complex
'
'   Ritorna il modulo di z al quadrato in formato complesso:
'
    CModQ = CCmp(z.Re * z.Re + z.Im * z.Im, 0#)
'
'
'
End Function
Private Function DATAN2(ByVal Y As Double, ByVal X As Double) As Double
'
'   Ritorna il valore dell' ArcoTangente di y/x
'   come implementata dal FORTRAN.
'   E':    -PI < DATAN2 <= PI.
'
    Select Case X
    Case Is > 0#
        DATAN2 = Atn(Y / X)
'
    Case Is < 0#
        If (0# <= Y) Then
            DATAN2 = Atn(Y / X) + PI
        ElseIf (Y < 0#) Then
            DATAN2 = Atn(Y / X) - PI
        End If
'
    Case Is = 0#
        DATAN2 = Sgn(Y) * PI_2
    End Select
'
'
'
End Function
Public Function CArg(z As Complex) As Double
'
'   Ritorna l' argomento del numero complesso z:
'
    CArg = DATAN2(z.Im, z.Re)
'
'
'
End Function
Public Function CAbs(z As Complex) As Double
'
'   Ritorna il valore assoluto (modulo) del numero complesso z:
'
    Dim r_ab#
'
    On Error GoTo Abs_ERR
'
    ' Calcolo diretto:
    'CAbs = Sqr(z.re * z.re + z.im * z.im)
'
'   Metodo di calcolo consigliato da Numerical Recipes:
'                 /
'                 |  |a| * Sqr(1 + (b/a)^2)  per |a| >= |b|
'    |a + ib| =  <
'                 |  |b| * Sqr(1 + (a/b)^2)  per |a| < |b|
'                 \
    If (z.Re = 0) And (z.Im = 0) Then
        CAbs = 0#
    ElseIf (Abs(z.Im) <= Abs(z.Re)) Then
        r_ab = z.Im / z.Re
        CAbs = Abs(z.Re) * Sqr(1# + r_ab * r_ab)
    Else
        r_ab = z.Re / z.Im
        CAbs = Abs(z.Im) * Sqr(1# + r_ab * r_ab)
    End If
'
'
Abs_ERR:
    If (Err.Number <> 0) Then
        CAbs = 0#
        Err.Clear
    End If
'
'
'
End Function
Public Sub CMsgBox(z As Complex, Optional ByVal Txt$ = "")
'
'
    MsgBox CFormat$(z) & Txt$
'
'
'
End Sub
Public Function CPtN(z As Complex, ByVal N As Long) As Complex
'
'   Ritorna il valore, complesso, di z elevato alla N,
'   con N reale ed intero:
'
    Dim Rho_N#, Theta_N#
'
    Rho_N = CAbs(z) ^ N
    Theta_N = N * DATAN2(z.Im, z.Re)
'
    CPtN.Re = Rho_N * Cos(Theta_N)
    CPtN.Im = Rho_N * Sin(Theta_N)
'
'
'
End Function
Public Function CDif(z1 As Complex, z2 As Complex) As Complex
'
'   Ritorna la differenza, complessa, dei due numeri complessi z1 - z2:
'
    CDif.Re = z1.Re - z2.Re
    CDif.Im = z1.Im - z2.Im
'
'
'
End Function
Public Function CSom(z1 As Complex, z2 As Complex) As Complex
'
'   Ritorna la somma, complessa, dei due numeri complessi z1 + z2:
'
    CSom.Re = z1.Re + z2.Re
    CSom.Im = z1.Im + z2.Im
'
'
'
End Function
Public Function CCmp(ByVal Re As Double, ByVal Im As Double) As Complex
'
'   Ritorna un numero complesso con parte reale re
'   e parte immaginaria im:
'
    CCmp.Re = Re
    CCmp.Im = Im
'
'
'
End Function
Public Function CMol(z1 As Complex, z2 As Complex) As Complex
'
'   Ritorna il prodotto, complesso, dei due numeri complessi z1 * z2:
'
    CMol.Re = z1.Re * z2.Re - z1.Im * z2.Im
    CMol.Im = z1.Re * z2.Im + z1.Im * z2.Re
'
'   Metodo di calcolo consigliato da Numerical Recipes:
'    (a + ib)(c + id) =(ac - bd) +i[(a + b)(c + d) - ac - bd]
'
'
'
End Function
Public Function CDiv(z1 As Complex, z2 As Complex, _
    Optional ByRef bErrore As Boolean) As Complex
'
'   Ritorna il rapporto, complesso, dei due numeri complessi z1 / z2:
'
    Dim r_dd1#, r_dd2#
'
    On Error GoTo CDiv_ERR
'
    ' Calcolo diretto:
    'r_dd2 = z2.re * z2.re + z2.im * z2.im
    'CDiv.re = (z1.re * z2.re + z1.im * z2.im) / r_dd2
    'CDiv.im = (z1.im * z2.re - z1.re * z2.im) / r_dd2
'
'   Metodo di calcolo consigliato da Numerical Recipes:
'               /  [a + b(d/c)] + i[b - a(d/c)]
'               |  ----------------------------   per |c| >= |d|
'    a + ib     |            c + d(d/c)
'    ------ =  <
'    c + id     |  [a(c/d) + b] + i[b(c/d) - a]
'               |  ----------------------------   per |c| < |d|
'               \            c(c/d) + d
'
    If (Abs(z2.Im) <= Abs(z2.Re)) Then
        r_dd1 = z2.Im / z2.Re
        r_dd2 = z2.Re + z2.Im * r_dd1
        CDiv.Re = (z1.Re + z1.Im * r_dd1) / r_dd2
        CDiv.Im = (z1.Im - z1.Re * r_dd1) / r_dd2
'
    Else
        r_dd1 = z2.Re / z2.Im
        r_dd2 = z2.Im + z2.Re * r_dd1
        CDiv.Re = (z1.Im + z1.Re * r_dd1) / r_dd2
        CDiv.Im = (z1.Im * r_dd1 - z1.Re) / r_dd2
    End If
'
'
CDiv_ERR:
    bErrore = (Err.Number <> 0)
    If bErrore Then
        'MsgBox Err.Description, vbCritical, " Funzione CDiv"
        CDiv.Re = CAbs(z1) * (Cos(CArg(z1) - CArg(z2))) / SMALNO
        CDiv.Im = CAbs(z1) * (Sin(CArg(z1) - CArg(z2))) / SMALNO
        Err.Clear
    End If
'
'
'
End Function
Public Function CCon(z As Complex) As Complex
'
'   Ritorna il coniugato del numero complesso z:
'
    CCon.Re = z.Re
    CCon.Im = -z.Im
'
'
'
End Function
Public Function CExp(z As Complex, Optional ByRef bErrore As Boolean) As Complex
'
'   Ritorna il valore, complesso, dell' esponenziale di z:
'
    Dim r_ex#
'
    On Error GoTo CExp_ERR
'
    r_ex = Exp(z.Re)
    CExp.Re = r_ex * Cos(z.Im)
    CExp.Im = r_ex * Sin(z.Im)
'
'
CExp_ERR:
    bErrore = (Err.Number <> 0)
    If bErrore Then
        MsgBox Err.Description, vbCritical, " Funzione CExp"
        r_ex = 1# / SMALNO
        CExp.Re = r_ex * Cos(z.Im)
        CExp.Im = r_ex * Sin(z.Im)
        Err.Clear
    End If
'
'
'
End Function

Public Function CEgu(z1 As Complex, z2 As Complex, _
    Optional ByRef d_re As Double, Optional ByRef d_im As Double) As Boolean
'
'   Verifica l' eguaglianza dei due numeri complessi z1 e z2.
'   Ritorna anche le differenze fra le parti reali e fra le
'   parti immaginarie:
'
    CEgu = (z1.Re = z2.Re) And (z1.Im = z2.Im)
    d_re = z1.Re - z2.Re
    d_im = z1.Im - z2.Im
'
'
'
End Function
Public Function CSqr(z As Complex, Optional ByRef bErrore As Boolean) As Complex
'
'   Ritorna il valore principale, complesso,
'   della radice quadrata di z:
'
    On Error GoTo CSqr_ERR
'
    ' Calcolo diretto:
    'Dim r_sq#, ar_2#
'
    'r_sq = Sqr(CAbs(z))
    'ar_2 = CArg(z) / 2#
'
    'CSqr.Re = r_sq * Cos(ar_2)
    'CSqr.Im = r_sq * Sin(ar_2)
'
'   Metodo di calcolo consigliato da Numerical Recipes:
'                    / 0                    per r = 0
'                    |
'                    | r + i(d/(2*r))       per r <> 0, c >= 0
'    Sqr(c + id) =  <
'                    | |d|/(2*r) + iw       per r <> 0, c < 0, d >= 0
'                    |
'                    \ |d|/(2*r) - iw       per r <> 0, c < 0, d < 0
'    con:
'          / 0                                              per c = d = 0
'          |
'    r =  <  Sqr(|c|) * Sqr((1 + Sqr(1 + (d/c)^2))/2)       per |c| >= |d|
'          |
'          \ Sqr(|d|) * Sqr((|c/d| + Sqr(1 + (c/d)^2))/2)   per |c| < |d|
'
    Dim r#
'
    ' Calcolo r:
    If (z.Re = 0#) And (z.Im = 0#) Then
        CSqr.Re = 0#
        CSqr.Im = 0#
        Exit Function
'
    ElseIf Abs(z.Im) <= Abs(z.Re) Then
        r = Sqr(Abs(z.Re)) * Sqr((1# + Sqr(1# + (z.Im / z.Re) ^ 2#)) / 2#)
'
    Else
        r = Sqr(Abs(z.Im)) * Sqr((Abs(z.Re / z.Im) + Sqr(1# + (z.Re / z.Im) ^ 2)) / 2#)
    End If
'
    ' Calcolo la radice:
    If (0# <= z.Re) Then
        CSqr.Re = r
        CSqr.Im = z.Im / (2# * r)
'
    ElseIf (0# <= z.Im) Then
        CSqr.Re = Abs(z.Im) / (2# * r)
        CSqr.Im = r
'
    Else
        CSqr.Re = Abs(z.Im) / (2# * r)
        CSqr.Im = -r
    End If
'
'
CSqr_ERR:
    bErrore = (Err.Number <> 0)
    If bErrore Then
        MsgBox Err.Description, vbCritical, " Funzione CSqr"
        CSqr.Re = 0#
        CSqr.Im = 0#
        Err.Clear
    End If
'
'
'
End Function
Public Function ReaC(z As Complex) As Double
'
'   Ritorna il valore della parte reale di z:
'
    ReaC = z.Re
'
'
'
End Function
Public Function ImmC(z As Complex) As Double
'
'   Ritorna il valore della parte immaginaria di z:
'
    ImmC = z.Im
'
'
'
End Function
Public Function CFormat$(z As Complex, _
    Optional ByVal strF$ = "", Optional ByVal NN As Long = 0, _
    Optional ByVal bCoCo As Boolean = False, _
    Optional ByVal sImmP As String = "j")
'
'   Ritorna una stringa con la rappresentazione del
'   numero complesso z.
'   Parametri opzionali:
'    strF$: stringa di formato. Segue la notazione standard
'           e viene applicata ugualmente alla parte reale ed
'           a quella immaginaria. Se strF$ = "" o e' assente,
'           il numero viene rappresentato con la massima
'           precisione possibile.
'
'    Nn:    se > 0 vengono rappresentati solo gli Nn digits
'           piu' significativi. Se Nn non e' sufficiente a
'           rappresentare almeno la parte intera del numero,
'           viene prima tentata la notazione scientifica e
'           poi viene segnalato un errore.
'
'    bCoCo: se True rappresenta una coppia di valori complessi
'           coniugati, i.e. con la parte immaginaria preceduta
'           dal simbolo .
'
'    sImmP: prefisso della parte immaginaria (consigliati i o j).
'
    Dim Rea$, Imm$
'
    Rea$ = Trim$(Format$(Abs(z.Re), strF$))
    Imm$ = Trim$(Format$(Abs(z.Im), strF$))
'
    If (0 < NN) Then
        If (InStr(Rea$, ".") - 1 <= NN) Then
            Rea$ = Left$(Rea$, NN)
        ElseIf (8 <= NN) Then
            Rea$ = Format$(Abs(z.Re), "0.00E+00")
        Else
            Rea$ = String$(NN, "*")
        End If
'
        If (InStr(Imm$, ".") - 1 <= NN) Then
            Imm$ = Left$(Imm$, NN)
        ElseIf (8 <= NN) Then
            Imm$ = Format$(Abs(z.Im), "0.00E+00")
        Else
            Imm$ = String$(NN, "*")
        End If
    End If
'
    If (z.Re < 0#) Then
        Rea$ = "-" & Rea$
    Else
        Rea$ = "+" & Rea$
    End If
'
    If bCoCo Then
        CFormat$ = Rea$ & "  " & sImmP & Imm$
    ElseIf (z.Im < 0#) Then
        CFormat$ = Rea$ & " - " & sImmP & Imm$
    Else
        CFormat$ = Rea$ & " + " & sImmP & Imm$
    End If
'
'
'
End Function
Public Sub CNnr(z As Complex, ByVal N As Long, wRt() As Complex)
'
'   Calcola le N radici Nesime del numero complesso z
'   con N > 0.
'   Le radici vengono ritornate nel vettore wRt(1 To N):
'
    Dim J&, Rho_N#, Theta#, Theta_N#, Phi_N#
    ReDim wRt(1 To N)
'
    Rho_N = (CAbs(z)) ^ (1# / CDbl(N))
    Theta_N = CArg(z) / CDbl(N)
    Phi_N = 2# * PI / CDbl(N)
'
    For J = 1 To N
        Theta = Theta_N + CDbl(J - 1) * Phi_N
        wRt(J).Re = Rho_N * Cos(Theta)
        wRt(J).Im = Rho_N * Sin(Theta)
    Next J
'
'
'
End Sub
Public Function CLog(z As Complex, Optional ByRef bErrore As Boolean) As Complex
'
'   Ritorna il valore principale, complesso,
'   del logaritmo naturale di z:
'
    On Error GoTo CLog_ERR
'
    CLog.Re = Log(CAbs(z))
    CLog.Im = CArg(z)
'
'
CLog_ERR:
    bErrore = (Err.Number <> 0)
    If bErrore Then
        MsgBox Err.Description, vbCritical, " Funzione CLog"
        CLog.Re = Log(SMALNO)
        CLog.Im = CArg(z)
        Err.Clear
    End If
'
'
'
End Function
Public Function CSin(z As Complex) As Complex
'
'   Ritorna il valore, complesso, del seno di z:
'
    Dim w_esp As Complex, w_esn As Complex
'
    w_esp = CMol(CCmp(0#, 1#), z)
    w_esn = CMol(CCmp(0#, -1#), z)
'
    CSin = CDiv(CDif(CExp(w_esp), CExp(w_esn)), CCmp(0#, 2#))
'
'
'
End Function
Public Function CCos(z As Complex) As Complex
'
'   Ritorna il valore, complesso, del coseno di z:
'
    Dim w_esp As Complex, w_esn As Complex
'
    w_esp = CMol(CCmp(0#, 1#), z)
    w_esn = CMol(CCmp(0#, -1#), z)
'
    CCos = CDiv(CSom(CExp(w_esp), CExp(w_esn)), CCmp(2#, 0#))
'
'
'
End Function
Public Function CTan(z As Complex) As Complex
'
'   Ritorna il valore, complesso, della tangente di z:
'
    CTan = CDiv(CSin(z), CCos(z))
'
'
'
End Function
Public Function CAtn(z As Complex) As Complex
'
'   Ritorna il valore, complesso, dell' arcotangente di z:
'
    Dim w_de As Complex, w_nu As Complex
'
    w_nu = CSom(CCmp(1#, 0#), CMol(CCmp(0#, 1), z))
    w_de = CDif(CCmp(1#, 0#), CMol(CCmp(0#, 1), z))
'
    CAtn = CDiv(CLog(CDiv(w_nu, w_de)), CCmp(0#, 2#))
'
'
'
End Function
Public Function CPtc(z As Complex, Alfa As Complex) As Complex
'
'   Ritorna il valore principale, complesso, di z elevato ad Alfa,
'   entrambi complessi:
'
    CPtc = CExp(CMol(Alfa, CLog(z)))
'
'
'
End Function
Public Function CSinh(z As Complex) As Complex
'
'   Ritorna il valore, complesso, del seno
'   iperbolico di z:
'
    Dim w_esn As Complex
'
    w_esn = CMol(CCmp(-1#, 0#), z)
'
    CSinh = CDiv(CDif(CExp(z), CExp(w_esn)), CCmp(2#, 0#))
'
'
'
End Function
Public Function CCosh(z As Complex) As Complex
'
'   Ritorna il valore, complesso, del coseno
'   iperbolico di z:
'
    Dim w_esn As Complex
'
    w_esn = CMol(CCmp(-1#, 0#), z)
'
    CCosh = CDiv(CSom(CExp(z), CExp(w_esn)), CCmp(2#, 0#))
'
'
'
End Function
Public Function CTanh(z As Complex) As Complex
'
'   Ritorna il valore, complesso, della tangente
'   iperbolica di z:
'
    CTanh = CDiv(CSinh(z), CCosh(z))
'
'
'
End Function
Public Function CAtnh(z As Complex) As Complex
'
'   Ritorna il valore, complesso, dell' arcotangente
'   iperbolica di z:
'
    Dim w_de As Complex, w_nu As Complex
'
    w_nu = CSom(CCmp(1#, 0#), z)
    w_de = CDif(CCmp(1#, 0#), z)
'
    CAtnh = CDiv(CLog(CDiv(w_nu, w_de)), CCmp(2#, 0#))
'
'
'
End Function
Public Function CAsin(z As Complex) As Complex
'
'   Ritorna il valore, complesso, dell' arcoseno di z:
'
    Dim w_as As Complex
'
    w_as = CSqr(CDif(CCmp(1#, 0#), CMol(z, z)))
    w_as = CSom(CMol(CCmp(0#, 1#), z), w_as)
'
    CAsin = CDiv(CLog(w_as), CCmp(0#, 1#))
'
'
'
End Function
Public Function CAcos(z As Complex) As Complex
'
'   Ritorna il valore, complesso, dell' arcocoseno di z:
'
    Dim w_ac As Complex
'
    w_ac = CSqr(CDif(CMol(z, z), CCmp(1#, 0#)))
    w_ac = CSom(z, w_ac)
'
    CAcos = CDiv(CLog(w_ac), CCmp(0#, 1#))
'
'
'
End Function
Public Function CAsinh(z As Complex) As Complex
'
'   Ritorna il valore, complesso, dell' arcoseno
'   iperbolico di z:
'
    Dim w_as As Complex
'
    w_as = CSqr(CSom(CMol(z, z), CCmp(1#, 0#)))
    w_as = CSom(z, w_as)
'
    CAsinh = CLog(w_as)
'
'
'
End Function
Public Function CAcosh(z As Complex) As Complex
'
'   Ritorna il valore, complesso, dell' arcocoseno
'   iperbolico di z:
'
    Dim w_ac As Complex
'
    w_ac = CSqr(CDif(CMol(z, z), CCmp(1#, 0#)))
    w_ac = CSom(z, w_ac)
'
    CAcosh = CLog(w_ac)
'
'
'
End Function

Public Sub CSWAP(z1 As Complex, z2 As Complex)
'
'
    Dim cTn As Complex
'
    cTn = z1
    z1 = z2
    z2 = cTn
'
'
'
End Sub
