Attribute VB_Name = "modHtFitter"
'=======================================================================================
' Descrizione.....: Routines per la ricerca di poli e zeri (MDOF)
'                   e per operazioni su funzioni di trasferimento.
' Nome dei Files..: modHtFitter.bas, modComplex.bas
'                   modMatrici.bas, modRPOLY.
' Data............: 27/09/2008
' Versione........: 1.0 a 32 bits.
' Sistema.........: VB6 (SP5) sotto Windows XP (SP2).
' Scritto da......: F. Languasco 
' E-Mail..........: MC7061@mclink.it
' DownLoads a.....: http://www.flanguasco.org
'=======================================================================================
'
'   La routine CalcolaAB_NormPoli per l' "Ht(f) fitting" e' derivata dall' articolo:
'    Parameter Estimation from Frequency Response Measurements
'    Using Rational Fraction Polynomials.
'    Mark H. Richardson & David L. Formenti
'    http://www.altracustica.org/docs/paper07.pdf
'
'   Ovunque necessario ho indicato con:
'    Ht(p) la funzione di trasferimento (p = Sigma + j * Omega);
'    Ht(f) la funzione di risposta in frequenza, i.e. l' Ht(p) calcolata a Sigma = 0.
'
Option Explicit
'
Public Type Ht_Type
    Fr As Double    ' Frequenza [Hz].
    Ht As Complex   ' Funzione di risposta in frequenza a Fr.
End Type
Private Sub OrdinaRadici(dModR() As Double, ByRef Radici() As Complex, _
    ByVal Low As Long, ByVal High As Long, Optional ByVal Verso As Long = -1)
'
'   Ordina il vettore Radici() nel verso della frequenza
'   naturale di risonanza crescente. Usa una versione
'   modificata dell' algoritmo Quick Sort.
'
'   Routine QuickSort2V:
'    dModR():   vettore che si vuole ordinare.
'    Radici():  vettore associato.
'    Low:       posizione iniziale della zona da ordinare.
'    High:      posizione finale della zona da ordinare.
'    Verso:     direzione dell' ordinamento:
'                > 0 -> dal minore al maggiore.
'                = 0 -> nessun ordinamento.
'                < 0 -> dal maggiore al minore.
'
    Dim RandIndex&, I&, J&, M$
    Dim dValTemp As Double  ' Tipo del vettore che si vuole ordinare.
    Dim cValTemp As Complex ' Tipo del vettore associato che si vuole ordinare.
    Dim Part As Double      ' Tipo della chiave di ordinamento.
'
    On Error GoTo OrdinaRadici_ERR
'
    If (Low < High) Then
        If (High - Low = 1) Then
            ' Only two elements in this subdivision; swap them
            ' if they are out of order, then end recursive calls:
            If ((0 < Verso) And (dModR(High) < dModR(Low))) _
            Or ((Verso < 0) And (dModR(Low) < dModR(High))) Then
                ' Vettore principale:
                'DSWAP dModR(Low), dModR(High)
                dValTemp = dModR(Low)
                dModR(Low) = dModR(High)
                dModR(High) = dValTemp
                ' Primo vettore associato:
                'CWAP Radici(Low), Radici(High)
                cValTemp = Radici(Low)
                Radici(Low) = Radici(High)
                Radici(High) = cValTemp
            End If
'
        Else
            ' Pick a pivot element, then move it to the end:
            RandIndex = (High + Low) / 2
            ' Vettore principale:
            'DSWAP dModR(High), dModR(RandIndex)
            dValTemp = dModR(High)
            dModR(High) = dModR(RandIndex)
            dModR(RandIndex) = dValTemp
            Part = dModR(High)
            ' Primo vettore associato:
            'CWAP Radici(High), Radici(RandIndex)
            cValTemp = Radici(High)
            Radici(High) = Radici(RandIndex)
            Radici(RandIndex) = cValTemp
'
            ' Move in from both sides towards the pivot element:
            Do
                I = Low: J = High
                Do While ((0 < Verso) And (I < J) And (dModR(I) <= Part)) _
                Or ((Verso < 0) And (I < J) And (dModR(I) >= Part))
                    I = I + 1
                Loop
                Do While ((0 < Verso) And (I < J) And (Part <= dModR(J))) _
                Or ((Verso < 0) And (I < J) And (dModR(J) <= Part))
                    J = J - 1
                Loop
'
                If (I < J) Then
                    ' We haven't reached the pivot element; it means that two
                    ' elements on either side are out of order, so swap them:
                    ' Vettore principale:
                    'DSWAP dModR(I), dModR(J)
                    dValTemp = dModR(I)
                    dModR(I) = dModR(J)
                    dModR(J) = dValTemp
                    ' Primo vettore associato:
                    'CWAP Radici(I), Radici(J)
                    cValTemp = Radici(I)
                    Radici(I) = Radici(J)
                    Radici(J) = cValTemp
                End If
'
            Loop While (I < J)
            ' Move the pivot element back to its proper place in the array:
            ' Vettore principale:
            'DSWAP dModR(I), dModR(High)
            dValTemp = dModR(I)
            dModR(I) = dModR(High)
            dModR(High) = dValTemp
            ' Primo vettore associato:
            'CWAP Radici(I), Radici(High)
            cValTemp = Radici(I)
            Radici(I) = Radici(High)
            Radici(High) = cValTemp
'
            ' Recursively call the QuickSort2V procedure (pass the smaller
            ' subdivision first to use less stack space):
            If ((I - Low) < (High - I)) Then
                OrdinaRadici dModR(), Radici(), Low, I - 1, Verso
                OrdinaRadici dModR(), Radici(), I + 1, High, Verso
            Else
                OrdinaRadici dModR(), Radici(), I + 1, High, Verso
                OrdinaRadici dModR(), Radici(), Low, I - 1, Verso
            End If
        End If
    End If
'
'
OrdinaRadici_ERR:
    If (Err.Number <> 0) Then
        M$ = "Errore " & Str$(Err.Number) & vbNewLine
        M$ = M$ & Err.Description
        MsgBox M$, vbCritical, " OrdinaRadici"
    End If
'
'
'
End Sub
Public Sub CalcolaHt_AB(AA() As Double, BB() As Double, Ht() As Ht_Type)
'
'   Calcola i valori della funzione di risposta in frequenza
'   complessa Ht().Ht usando i coefficienti, reali, AA() del
'   numeratore e BB() del denominatore.
'   I valori di frequenza da usare devono essere passati
'   in Ht().Fr:
'
'                 M
'                Som(AA(K) * (j * 2 * PI * Ht(f).Fr) ^ K)
'                K=0
'   Ht(f).Ht = --------------------------------------------
'                 N
'                Som(BB(K) * (j * 2 * PI * Ht(f).Fr) ^ K)
'                K=0
'
    Dim K&, NF&, L&, M&, N&
    Dim COmN As Complex, CNum As Complex, CDen As Complex
'
    NF = UBound(Ht)
    M = UBound(AA)
    N = UBound(BB)
'
    For L = 1 To NF
        'Ht(L).Ht = CalcolaHt_ABp(AA(), BB(), CCmp(0#, PI2 * Ht(L).Fr))
'
        CNum = CCmp(0#, 0#)
        For K = 0 To M
            COmN = CPtN(CCmp(0#, PI2 * Ht(L).Fr), K)
            CNum = CSom(CNum, CMol(CCmp(AA(K), 0#), COmN))
        Next K
'
        CDen = CCmp(0#, 0#)
        For K = 0 To N
            COmN = CPtN(CCmp(0#, PI2 * Ht(L).Fr), K)
            CDen = CSom(CDen, CMol(CCmp(BB(K), 0#), COmN))
        Next K
'
        Ht(L).Ht = CDiv(CNum, CDen)
    Next L
'
'
'
End Sub
Public Sub CalcolaHt_ZP(ByVal Kg As Double, Zeri() As Complex, Poli() As Complex, _
     Ht() As Ht_Type)
'
'   Calcola i valori della funzione di risposta in frequenza
'   complessa Ht().Ht usando Kg ed i valori, complessi, di
'   Zeri() e Poli() in rad/s.
'   I valori di frequenza da usare devono essere passati
'   in Ht().Fr:
'
'                      M
'                     Prod((j * 2 * PI * Ht(f).Fr) - Zeri(K))
'                     K=0
'   Ht(f).Ht = Kg * -------------------------------------------
'                      N
'                     Prod((j * 2 * PI * Ht(f).Fr) - Poli(K))
'                     K=0
'
    Dim K&, L&, NF&, M&, N&
    Dim CNum As Complex, CDen As Complex
'
    NF = UBound(Ht)
    M = UBound(Zeri)
    N = UBound(Poli)
'
    For L = 1 To NF
        'Ht(L).Ht = CalcolaHt_ZPp(Kg, Zeri(), Poli(), CCmp(0#, PI2 * Ht(L).Fr))
'
        CNum = CCmp(Kg, 0#)
        For K = 1 To M
            CNum = CMol(CNum, CDif(CCmp(0#, PI2 * Ht(L).Fr), Zeri(K)))
        Next K
'
        CDen = CCmp(1#, 0#)
        For K = 1 To N
            CDen = CMol(CDen, CDif(CCmp(0#, PI2 * Ht(L).Fr), Poli(K)))
        Next K
'
        Ht(L).Ht = CDiv(CNum, CDen)
    Next L
'
'
'
End Sub
Public Function CalcolaHt_ABp(AA() As Double, BB() As Double, p As Complex) As Complex
'
'   Calcola il valore complesso della funzione di trasferimento Ht(p),
'   alla pulsazione complessa p = Sigma + j * Omega, usando i coef-
'   ficienti, reali, AA() del numeratore e BB() del denominatore;
'   Sigma e Omega sono in rad/s:
'
'              M
'             Som(AA(K) * p ^ K)
'             K=0
'   Ht(p) = ----------------------
'              N
'             Som(BB(K) * p ^ K)
'             K=0
'
    Dim K&, M&, N&
    Dim COmN As Complex, CNum As Complex, CDen As Complex
'
    M = UBound(AA)
    N = UBound(BB)
'
    CNum = CCmp(0#, 0#)
    For K = 0 To M
        COmN = CPtN(p, K)
        CNum = CSom(CNum, CMol(CCmp(AA(K), 0#), COmN))
    Next K
'
    CDen = CCmp(0#, 0#)
    For K = 0 To N
        COmN = CPtN(p, K)
        CDen = CSom(CDen, CMol(CCmp(BB(K), 0#), COmN))
    Next K
'
    CalcolaHt_ABp = CDiv(CNum, CDen)
'
'
'
End Function
Public Function CalcolaHt_ZPp(ByVal Kg As Double, Zeri() As Complex, Poli() As Complex, _
    p As Complex) As Complex
'
'   Calcola il valore complesso della funzione di trasferimento Ht(p),
'   alla pulsazione complessa p = Sigma + j * Omega, usando Kg, Zeri()
'   e Poli(); Sigma, Omega, Zeri() e Poli() sono in rad/s:
'
'                   M
'                  Prod(p - Zeri(K))
'                  K=1
'   Ht(p) = Kg * ---------------------
'                   N
'                  Prod(p - Poli(K))
'                  K=1
'
    Dim K&, M&, N&
    Dim CNum As Complex, CDen As Complex
'
    M = UBound(Zeri)
    N = UBound(Poli)
'
    CNum = CCmp(Kg, 0#)
    For K = 1 To M
        CNum = CMol(CNum, CDif(p, Zeri(K)))
    Next K
'
    CDen = CCmp(1#, 0#)
    For K = 1 To N
        CDen = CMol(CDen, CDif(p, Poli(K)))
    Next K
'
    CalcolaHt_ZPp = CDiv(CNum, CDen)
'
'
'
End Function
Public Function CalcolaZeriPoli(AA() As Double, BB() As Double, _
    ByRef Kg As Double, ByRef Zeri() As Complex, ByRef Poli() As Complex) As Boolean
'
'   Calcola il guadagno Kg, gli Zeri() ed i Poli() della funzione di
'   trasferimento, partendo dai coefficienti AA() e BB() dei polinomi
'   a numeratore e denominatore:
'
    Dim I&, K&, Nn1&, Nd1&, KgN#, KgD#, bFailZ As Boolean, bFailP As Boolean
    Dim Aa1#()  ' Vettore dei coefficienti, reali, del numeratore del polinomio.
    Dim Bb1#()  ' Vettore dei coefficienti, reali, del denominatore del polinomio.
    Dim ZeriRe#(), ZeriIm#(), PoliRe#(), PoliIm#()
'
    ' Sistemo i coefficienti come
    ' richiesto dalla routine RPOLY:
    Nn1 = UBound(AA)
    ReDim Aa1(1 To Nn1 + 1)
    KgN = AA(Nn1)
    For I = 0 To Nn1
        Aa1(I + 1) = AA(I) / KgN
    Next I
'
    Nd1 = UBound(BB)
    ReDim Bb1(1 To Nd1 + 1)
    KgD = BB(Nd1)
    For K = 0 To Nd1
        Bb1(K + 1) = BB(K) / KgD
    Next K
'
    If (1 <= Nn1) Then
        ' Calcolo gli zeri:
        RPOLY Aa1(), ZeriRe(), ZeriIm(), , bFailZ
        ReDim Zeri(0 To UBound(ZeriRe))
        For K = 1 To UBound(ZeriRe)
            Zeri(K) = CCmp(ZeriRe(K), ZeriIm(K))
        Next K
    Else
        ReDim Zeri(0 To 0)
    End If
'
    If (1 <= Nd1) Then
        ' Calcolo i poli:
        RPOLY Bb1(), PoliRe(), PoliIm(), , bFailP
        ReDim Poli(0 To UBound(PoliRe))
        For K = 1 To UBound(PoliRe)
            Poli(K) = CCmp(PoliRe(K), PoliIm(K))
        Next K
    Else
        ReDim Poli(0 To 0)
    End If
'
    ' Calcolo i guadagni dei polinomi
    ' espressi in forma normale:
    Zeri(0) = CCmp(KgN, 0#)
    Poli(0) = CCmp(KgD, 0#)
    Kg = KgN / KgD
'
'
CalcolaZeriPoli_ERR:
    CalcolaZeriPoli = (Err.Number = 0)
'
    If bFailZ Then
        MsgBox "Errore nella ricerca degli zeri:" & vbNewLine & _
               "provare a cambiare il grado" & vbNewLine & "del numeratore.", _
               vbExclamation, " modHFitter: CalcolaZeriPoli"
        CalcolaZeriPoli = False
    End If
'
    If bFailP Then
        MsgBox "Errore nella ricerca dei poli:" & vbNewLine & _
               "provare a cambiare il grado" & vbNewLine & "del denominatore.", _
               vbExclamation, " modHFitter: CalcolaZeriPoli"
        CalcolaZeriPoli = False
    End If
'
'
'
End Function
Public Function RiduciPolinomi(CC() As Double) As Double()
'
'   Ritorna un polinomio ridotto al grado piu' alto
'   di CC() con coefficiente diverso da zero:
'
    Dim K&, CC1#()
'
    On Error GoTo RiduciPolinomi_ERR
'
    CC1() = CC()
'
    For K = UBound(CC1) To 0 Step -1
        If (CC1(K) <> 0#) Then
            Exit For
        End If
    Next K
'
    ReDim Preserve CC1(0 To K)
    RiduciPolinomi = CC1()
'
'
RiduciPolinomi_ERR:
    If (Err.Number <> 0) Then
        Dim M$
        M$ = "Errore " & Str$(Err.Number) & vbNewLine _
           & Err.Description & vbNewLine & vbNewLine _
           & "I coefficienti del polinomio" & vbNewLine _
           & "sono tutti uguali a zero."
        MsgBox M$, vbCritical, " modHFitter: RiduciPolinomi"
    End If
'
'
'
End Function
Public Function ScartaRadici(Radici() As Complex, FMax As Double) As Complex()
'
'   Routine NON usata, in questa versione:
'
    Dim I&, K&, NK&, RadR() As Complex
'
    NK = UBound(Radici)
'
    I = 0
    For K = 1 To NK
        If (-FMax < Radici(K).Re / PI2) _
        And (Radici(K).Re / PI2 < FMax) Then
            I = I + 1
            ReDim Preserve RadR(1 To I)
            RadR(I) = Radici(K)
        End If
    Next K
'
    ScartaRadici = RadR()
'
'
'
End Function
Public Sub ClassificaRadici(ByRef Radici() As Complex, Radici_Cl() As Long)
'
'   Ritorna la matrice Radici_Cl() con una classifica, radice per radice,
'   della tipologia di Radici() secondo la tabella seguente:
'
'    Radici_Cl(1 To Nr, 1 To 2)     K = 1 To Nr -> N della radice
'
'    -------------------------------------------------------------
'   | K, 1 | K, 2 | Tipologia                                     |
'   |=============================================================|
'   |  1   |  n   | Radice Kma reale di molteplicita' n           |
'   |-------------------------------------------------------------|
'   |  2   |  n   | Radice Kma immaginaria di molteplicita' n (*) |
'   |-------------------------------------------------------------|
'   |  3   |  n   | Radice Kma complessa o immaginaria coniugata  |
'   |      |      | di molteplicita' n                            |
'   |-------------------------------------------------------------|
'   | ...  | ...  | ...                                           |
'   |-------------------------------------------------------------|
'   | -1   |  -   | Radice Kma non classificata perche' parte di  |
'   |      |      | un caso gia' condiderato precedentemente      |
'   |-------------------------------------------------------------|
'   | ...  | ...  | (le radici non classificate sono spostate     |
'   |      |      |  in fondo alla lista)                         |
'   |-------------------------------------------------------------|
'
'   Nota: a causa degli errori di arrotondamento, inevitabili
'   nei calcoli di ricerca delle radici, e' impossibile dire,
'   conoscendone solo il valore, se due radici siano o no
'   coincidenti. In questa routine si e' scelto di considerare
'   coincidenti radici con distanza < EpsRe sulla parte reale
'   e < EpsIm sulla parte immaginaria;
'    EpsRe = EpsC * Abs(Radici(K).Re),
'    EpsIm = EpsC * Abs(Radici(K).Im).
'
'   (*) caso non presente nelle radici di un polinomio a coef-
'       ficienti reali.
'
    Dim I&, K&, J&, Nr&, dModR#(), EpsRe#, EpsIm#
    Const EpsC# = 0.001, Eps# = 0.000001
'
    Nr = UBound(Radici)
    ReDim Radici_Cl(0 To Nr, 1 To 2)
'
    ' Ordino il vettore Radici() nel verso della
    ' frequenza naturale di risonanza crescente:
    ReDim dModR(0 To Nr)
    For I = 1 To Nr
        dModR(I) = CAbs(Radici(I))
    Next I
    OrdinaRadici dModR(), Radici(), 1, Nr, 1
'
    ' Classifico le radici in base
    ' alla loro molteplicita':
    For K = 1 To Nr
        If Radici_Cl(K, 1) = 0 Then ' La radice Kma non e' ancora
                                    ' stata classificata.
'
            Radici_Cl(K, 2) = 1     ' La radice Kma ha almeno
                                    ' molteplicita' 1.
'
            For J = K + 1 To Nr
                ' Calcolo la distanza della radice Kma dalla radice Jma:
                EpsRe = EpsC * Abs(Radici(K).Re)
                EpsIm = EpsC * Abs(Radici(K).Im)
                If (Abs(Radici(K).Re - Radici(J).Re) <= EpsRe) _
                And (Abs(Radici(K).Im - Radici(J).Im) <= EpsIm) Then
                    Radici_Cl(J, 1) = -1        ' La distanza e' < Eps e la radice
                                                ' Jma non verra' piu' considerata.
'
                    Radici_Cl(K, 2) = Radici_Cl(K, 2) + 1   ' La radice Kma ha
                                                            ' molteplicita'
                                                            ' Radici_Cl(K, 2).
                End If
            Next J
        End If
    Next K
'
    ' Classifico il tipo di radice:
    For K = 1 To Nr
        If (0 <= Radici_Cl(K, 1)) Then  ' La radice Kma e' classificata.
            If (Abs(Radici(K).Im) < Eps) Then
                ' La radice e' reale:
                Radici_Cl(K, 1) = 1
'
'            ElseIf (Abs(Radici(K).re) < eps) Then
'                ' La radice e' immaginaria:
'                Radici_Cl(K, 1) = 2
'
            Else
                ' La radice e' complessa o
                ' immaginaria coniugata ...:
                Radici_Cl(K, 1) = 3
'
                ' ... ed elimino dalla lista
                ' anche i suoi coniugati:
                For J = K + 1 To Nr
                    ' Calcolo la distanza della radice Kma dalla radice Jma:
                    EpsRe = EpsC * Abs(Radici(K).Re)
                    EpsIm = EpsC * Abs(Radici(K).Im)
                    If (Abs(Radici(K).Re - Radici(J).Re) <= EpsRe) _
                    And (Abs(Radici(K).Im + Radici(J).Im) <= EpsIm) Then
                        Radici_Cl(J, 1) = -1        ' La distanza e' < Eps, la radice
                                                    ' Jma e' la coniugata della Kma e
                                                    ' non verra' piu' considerata.
                    End If
                Next J
            End If
        End If
    Next K
'
    ' Sposto in fondo alla lista le radici non classificate:
    For K = 1 To Nr - 1
        For J = K + 1 To Nr
            If (Radici_Cl(K, 1) < 0) And (0 < Radici_Cl(J, 1)) Then
                ISWAP Radici_Cl(J, 1), Radici_Cl(K, 1)
                ISWAP Radici_Cl(J, 2), Radici_Cl(K, 2)
                CSWAP Radici(J), Radici(K)
                Exit For
            End If
        Next J
    Next K
'
'
'
End Sub
Public Function HtFitter_NormPoli(Ht() As Ht_Type, ByVal M As Long, ByVal N As Long, _
    ByRef AA() As Double, ByRef BB() As Double, _
    ByRef Kg As Double, ByRef Zeri() As Complex, ByRef Poli() As Complex, _
    ByRef Ht_Fit() As Ht_Type) As Boolean
'
'   Data la funzione di risposta in frequenza misurata Ht(f), questa
'   routine calcola una funzione di trasferimento Ht_Fit(p) interpolata
'   come rapporto di due polinomi aventi, rispettivamente grado M e coef-
'   ficienti AA() al numeratore e grado N e coefficienti BB() al
'   denominatore. Della funzione interpolata calcola pure gli Zeri()
'   i Poli() ed il coefficiente Kg.
'   Il metodo usato e' quello del calcolo diretto dei coefficienti
'   di polinomi normali.
'
'   La funzione ritorna False in caso di errori, e.g. quando il sistema
'   di equazioni da risolvere presenta una matrice singolare.
'
'              M
'             Som(AA(K) * p ^ K)
'             K=0
'   Ht(p) = ----------------------      con p = j * w;   w = 2 * PI * f
'              N
'             Som(BB(K) * p ^ K)
'             K=0
'
    Dim AAr#(), BBr#()
'
    On Error GoTo HtFitter_NormPoli_ERR
'
'-- Calcolo i coefficienti AA() e BB(): ------------------------------------------------
'
    If (Not CalcolaAB_NormPoli(Ht(), M, N, AA(), BB())) Then
        HtFitter_NormPoli = False
        Exit Function
    End If
'
'-- Calcolo Zeri() e Poli(): -----------------------------------------------------------
'
'    AAr() = AA()
'    BBr() = BB()
    AAr() = RiduciPolinomi(AA())
    BBr() = RiduciPolinomi(BB())
'
    If (Not CalcolaZeriPoli(AAr(), BBr(), Kg, Zeri(), Poli())) Then
        HtFitter_NormPoli = False
        Exit Function
    End If
'
'    Zeri() = ScartaRadici(Zeri(), Ht(NF).FV)
'    Poli() = ScartaRadici(Poli(), Ht(NF).FV)
'
'-- Calcolo la Funzione di Trasferimento interpolata: ----------------------------------
'
    Ht_Fit() = Ht() ' Copio in Ht_Fit() le frequenze di Ht().
'
    ' Dai coefficienti AA() e BB() ...:
    'CalcolaHt_AB AA(), BB(), Ht_Fit()
'
    ' ... o da Kg, Zeri() e Poli():
    CalcolaHt_ZP Kg, Zeri(), Poli(), Ht_Fit()
'
'
HtFitter_NormPoli_ERR:
    HtFitter_NormPoli = (Err.Number = 0)
'
    If (Err.Number <> 0) Then
        MsgBox Err.Description, vbCritical, " modHFitter: HtFitter_NormPoli"
    End If
'
'
'
End Function
Private Function CalcolaAB_NormPoli(Ht() As Ht_Type, ByVal M As Long, ByVal N As Long, _
    ByRef AA() As Double, ByRef BB() As Double) As Boolean
'
'   Calcola i coefficienti AA() e BB() dei polinomi dell' Ht(p)
'   interpolata con il metodo dei polinomi normali:
'
    Dim I&, J&, K&, L&, OmNorm#, jOm() As Complex
'
    Dim PP() As Complex, TT() As Complex, WW() As Complex
    Dim XX#(), XXT#(), YY#(), ZZ#(), GG#(), FF#(), CC#(), AB#()
'
    On Error GoTo CalcolaAB_NormPoli_ERR
'
    L = UBound(Ht)
'
    ' Normalizzo le pulsazioni
    ' nel campo 0 -> 1:
    ReDim jOm(1 To L)
    OmNorm = PI2 * Ht(L).Fr
    For I = 1 To L
        jOm(I) = CCmp(0#, Ht(I).Fr / Ht(L).Fr)
    Next I
'
'-- Preparo le matrici: ----------------------------------------------------------------
'
    ' Preparo la matrice PP():
    ReDim PP(1 To L, 1 To M + 1)
'
    For I = 1 To L
        For K = 1 To M + 1
            PP(I, K) = CPtN(jOm(I), K - 1)
        Next K
    Next I
'
    ' Preparo la matrice TT() completa:
    ReDim TT(1 To L, 1 To N + 1)
'
    For I = 1 To L
        For K = 1 To N + 1
            TT(I, K) = CMol(Ht(I).Ht, CPtN(jOm(I), K - 1))
        Next K
    Next I
'
    ' Il vettore WW() e' l' ultima
    ' colonna della matrice TT() completa:
    ReDim WW(1 To L)
    For I = 1 To L
        WW(I) = TT(I, N + 1)
    Next I
'
    ' Elimino l' ultima colonna
    ' della matrice TT():
    ReDim Preserve TT(1 To L, 1 To N)
'
                                                                     ' Dimensioni:
    XX() = ReCMScMol(CMMol(CTrasposta(CMConiugata(PP())), TT()), -1) ' XX(1 To M + 1, 1 To N)
'
    XXT() = Trasposta(XX())                                          ' XXT(1 To N, 1 To M + 1)
'
    YY() = ReCMScMol(CMMol(CTrasposta(CMConiugata(PP())), PP()))     ' YY(1 To M + 1, 1 To N)
'
    ZZ() = ReCMScMol(CMMol(CTrasposta(CMConiugata(TT())), TT()))     ' ZZ(1 To N, 1 To N)
'
    GG() = ReCVScMol(CMVMol(CTrasposta(CMConiugata(PP())), WW()))    ' GG(1 To M + 1)
'
    FF() = ReCVScMol(CMVMol(CTrasposta(CMConiugata(TT())), WW()))    ' FF(1 To M + 1)
'
    ' Preparo la matrice CC() dei coefficienti
    ' del sistema di equazioni da risolvere:
    ReDim CC(1 To M + 1 + N, 1 To M + 1 + N)
'
    For I = 1 To M + 1
        For K = 1 To M + 1                      '      | YY  :    |
            CC(I, K) = YY(I, K)                 ' CC = |..........|
        Next K                                  '      |     :    |
    Next I
'
    For I = 1 To M + 1
        For K = 1 To N                          '      | YY  : XX |
            CC(I, K + M + 1) = XX(I, K)         ' CC = |..........|
        Next K                                  '      |     :    |
    Next I
'
    For I = 1 To N
        For K = 1 To M + 1                      '      | YY  : XX |
            CC(I + M + 1, K) = XXT(I, K)        ' CC = |..........|
        Next K                                  '      | XXT :    |
    Next I
'
    For I = 1 To N
        For K = 1 To N                          '      | YY  : XX |
            CC(I + M + 1, K + M + 1) = ZZ(I, K) ' CC = |..........|
        Next K                                  '      | XXT : ZZ |
    Next I
'
    ' Preparo il vettore AB() dei termini noti;
    ' verranno sostituiti, in LinEqu, dalle soluzioni:
    ReDim AB(1 To M + 1 + N)
'
    For I = 1 To M + 1                          '      | GG |
        AB(I) = GG(I)                           ' AB = |....|
    Next I                                      '      |    |
'
    For I = 1 To N                              '      | GG |
        AB(I + M + 1) = FF(I)                   ' AB = |....|
    Next I                                      '      | FF |
'
'-- Calcolo, contemporanemente, i coefficienti AA() e BB() nel vettore AB(): -----------
'
    If (Not LinEqu(CC(), AB())) Then Err.Raise 1001, _
        "CalcolaAB_NormPoli", "Errore in LinEqu"
'
    ' Recupero, dal vettore AB(),
    ' i soli coefficienti AA() ...:
    ReDim AA(0 To M)
    For K = 1 To M + 1
        AA(K - 1) = AB(K)
    Next K
'
    ' ... e BB():
    ReDim BB(0 To N)
    For K = 1 To N
        BB(K - 1) = AB(M + 1 + K)
    Next K
    BB(N) = 1#
'
    ' Ricalcolo i coefficienti per
    ' la frequenza denormalizzata:
    For K = 0 To M
        AA(K) = AA(K) / (OmNorm ^ K)
    Next K
    For K = 0 To N
        BB(K) = BB(K) / (OmNorm ^ K)
    Next K
'
'
CalcolaAB_NormPoli_ERR:
    CalcolaAB_NormPoli = (Err.Number = 0)
'
    If (Err.Number <> 0) Then
        MsgBox Err.Description, vbCritical, " modHFitter: CalcolaAB_NormPoli"
    End If
'
'
'
End Function
Public Sub ZeriPoli2AB(ByVal Kg As Double, Zeri() As Complex, Poli() As Complex, _
    ByRef AA() As Double, ByRef BB() As Double)
'
'   Calcola, da Kg, Zeri() e Poli(), i coefficienti reali dei polinomi
'   a numeratore e denominatore della funzione di trasferimento:
'
    Dim I&, NZ&, NP&, RR() As Complex, RRc() As Complex
'
    ' Calcolo il polinomio al numeratore
    ' della funzione di trasferimento:
    NZ = UBound(Zeri)
'
    ReDim RR(1 To 2), RRc(1 To 1)
    RR(2) = CCmp(1#, 0#)
'
    RRc(1) = CCmp(Kg, 0#)
    For I = 1 To NZ
        RR(1) = CMol(CCmp(-1#, 0#), Zeri(I))
        RRc() = CPMPY(RRc(), RR())
    Next I
'
    ReDim AA(0 To NZ)
    For I = 0 To NZ
        AA(I) = RRc(I + 1).Re
    Next I
'
    ' Calcolo il polinomio al denominatore
    ' della funzione di trasferimento:
    NP = UBound(Poli)
'
    ReDim RR(1 To 2), RRc(1 To 1)
    RR(2) = CCmp(1#, 0#)
'
    RRc(1) = CCmp(1#, 0#)
    For I = 1 To NP
        RR(1) = CMol(CCmp(-1#, 0#), Poli(I))
        RRc() = CPMPY(RRc(), RR())
    Next I
'
    ReDim BB(0 To NP)
    For I = 0 To NP
        BB(I) = RRc(I + 1).Re
    Next I
'
'
'
End Sub
