Attribute VB_Name = "modMatrici"
'=======================================================================================
' Descrizione.....: Routines per calcolo matriciale.
' Nome dei Files..: modMatrici.bas, modComplex.bas
' Data............: 11/02/2000
' Aggiornamento...: 12/01/2001
' Aggiornamento...: 17/07/2002 (aggiunta DLLSQ)
' Aggiornamento...: 25/09/2008 (aggiunte le operazioni su matrici a valori complessi).
' Versione........: 2.0 a 32 bits.
' Sistema.........: VB6 (SP5) sotto Windows XP (SP2).
' Scritto da......: F. Languasco 
' E-Mail..........: MC7061@mclink.it
' DownLoads a.....: http://www.flanguasco.org
'=======================================================================================
'   Comprende le seguenti funzioni e subroutines:
'
'    c() = MSom(a(), b()):            ritorna la somma di due matrici (C).
'    c() = MDif(a(), b()):            ritorna la differenza di due matrici (C).
'    c() = MMol(a(), b()):            ritorna il prodotto di due matrici (C).
'    c() = MMSc(a(),s):               ritorna il prodotto di una matrice per uno scalare.
'    c() = MIde(N):                   ritorna una matrice identica di ordine N (C).
'    c() = VScMol(a(), [s]):          ritorna il prodotto di un vettore per uno scalare.
'    c() = Trasposta(a()):            ritorna la trasposta di una matrice (C).
'    B = MEgu(a(), b()):              verifica l' eguaglianza a() = b() (C).
'    B = Inversa(a(), c()):           calcola la matrice inversa (si)(*)(C).
'    B = Determinante(a(), det):      calcola il determinante di una matrice
'                                     quadrata (si)(*)(C).
'
'
'    Cc() = ReCMScMol(Ca(), [s]):     ritorna la parte reale della matrice Ca(),
'                                     a valori complessi, moltiplicata per uno scalare.
'    Cc() = CTrasposta(Ca()):         ritorna la trasposta di una matrice
'                                     a valori complessi.
'    Cc() = CMConiugata(Ca()):        ritorna la coniugata di una matrice
'                                     a valori complessi.
'
'    B = LinEqu(a(), b()):            calcola le soluzioni di un sistema di
'                                     equazioni lineari (si)(*)(C).
'    GAUS a(), b(), x(), IERROR:      calcola le soluzioni di un sistema di
'                                     equazioni lineari (**).
'    DLLSQ a(), b(), x(), IER, ...:   calcola, con il metodo dei minimi quadrati,
'                                     le soluzioni di un sistema di M equazioni lineari
'                                     in N incognite con M >= N.  Permette la soluzione
'                                     contemporanea di piu' sistemi aventi la stessa
'                                     matrice a() (***).
'    TRISOL a(), b(), c(), d():       risolve un sistema di equazioni tridiagonale
'                                     con il metodo di Thomas.
'    B = Jacobi (a(), d(), v()):      calcola gli autovalori e gli autovettori di una
'                                     matrice quadrata, reale e simmetrica (*).
'    B = AutoValori(a(), wr(), wi()): calcola gli autovalori, reali e complessi, di una
'                                     matrice quadrata e reale (*).
'    POWERM a(), v(), Lambda:         calcola l' autovalore dominante e l' autovettore
'                                     corrispondente, reali, di una matrice quadrata e
'                                     reale (**).
'    POWINV a(), v(), Alfa, Lambda:   calcola l' autovalore piu' vicino ad Alfa ed il
'                                     corrispondente autovettore, reali, di una matrice
'                                     quadrata e reale (**).
'
'   (si):   ritornano False se la matrice e' singolare.
'           Vedere l' ultima istruzione in ludcmp.
'
'   (C):    esiste anche la versione che opera su matrici a
'           valori complessi (e.g. CMsom, CInversa, etc...).
'
'   Tipi:   a() as Double, b() as Double, c() as Double, d() as Double
'           v(), wr() as Double, wi() as Double, x() as Double
'           Ca() as Complex, Cb() as Complex, Cc() as Complex, Cd() as Complex
'           d as Double, det as Double, s as Double, Alfa as Double, Lambda as Double
'           N as Long, IER as Long, IERROR as Long
'           B as Boolean
'
'   Note di traduzione: le matrici ed i vettori marcati ByRef nella dichiarazione
'                       iniziale delle routines vengono modificati dalle stesse.
'                       E' stata mantenuta, per quanto possibile, la struttura dei
'                       programmi originali: ne deriva l' uso intensivo dei GoTo
'                       (cosi' invisi ai puristi...).
'
'   Nota:   tutte le matrici ed i vettori usati da queste  routines
'           iniziano dall' indice 1: e.g. a(1 To N, 1 To N)
'
'   (*):    tratte da:  Numerical Recipes in Fortran 77.
'                       The Art of Scientific Computing.
'                       Second Edition.
'                       Volume 1 of Fortran Numerical Recipes.
'                       William H. Press, Saul A. Teukolsky,
'                       William T. Vetterling, Brian P. Flannery
'
'   (**):   tratte da:  Numerical Analysis.
'                       L. W. Johnson, R. D. Riess
'                       Addison-Wesley Publishing Company
'
'   (***):  tratta da:  PDP-10 Freeware Archives: 10-101 SSP, Version: 3, July 1973
'                       Author: Sandia Laboratories (I.B.M.)
'                       Revised by: H. David Todd, Wesleyan Univ., Middletown, CT
'                       Source Language: FORTRAN IV     (Vedi file DLLSQ.txt)
Option Explicit
Public Function CMVMol(Ca() As Complex, Cb() As Complex) As Complex()
'
'   Ritorna il prodotto della matrice a valori complessi Ca()
'   per il vettore a valori complessi Cb().
'   Il numero di colonne di Ca() deve essere uguale
'   al numero di righe di Cb():
'
    Dim I&, J&, K&
'
    On Error GoTo CMVMol_ERR
'
    If (UBound(Ca, 2) <> UBound(Cb)) Then Err.Raise 5
    ReDim C(1 To UBound(Ca, 1)) As Complex
'
    For I = 1 To UBound(Ca, 1)
        For J = 1 To UBound(Cb)
            C(I) = CSom(C(I), CMol(Ca(I, J), Cb(J)))
        Next J
    Next I
'
    CMVMol = C()
'
'
CMVMol_ERR:
    If (Err.Number <> 0) Then
        MsgBox "La matrice ed il vettore non hanno ordini compatibili.", vbCritical, " CMVMol"
    End If
'
'
'
End Function
Public Function MVMol(Ca() As Double, Cb() As Double) As Double()
'
'   Ritorna il prodotto della matrice a valori reali Ca()
'   per il vettore a valori reali Cb().
'   Il numero di colonne di Ca() deve essere uguale
'   al numero di righe di Cb():
'
    Dim I&, J&, K&
'
    On Error GoTo MVMol_ERR
'
    If (UBound(Ca, 2) <> UBound(Cb)) Then Err.Raise 5
    ReDim C#(1 To UBound(Ca, 1))
'
    For I = 1 To UBound(Ca, 1)
        For J = 1 To UBound(Cb)
            C(I) = C(I) + Ca(I, J) * Cb(J)
        Next J
    Next I
'
    MVMol = C()
'
'
MVMol_ERR:
    If (Err <> 0) Then
        MsgBox "La matrice ed il vettore non hanno ordini compatibili.", vbCritical, " MVMol"
    End If
'
'
'
End Function
Public Sub DLLSQ(a() As Double, B() As Double, ByRef X() As Double, _
    ByRef IER As Long, ByRef IPIV() As Long, ByRef AUX() As Double, _
    Optional ByVal Eps As Double = 0.000001)
'
'   PURPOSE
'   TO SOLVE LINEAR LEAST SQUARES PROBLEMS, I.E. TO MINIMIZE
'   THE EUCLIDEAN NORM OF B-A*X, WHERE A IS A M BY N MATRIX
'   WITH M NOT LESS THAN N. IN THE SPECIAL CASE M=N SYSTEMS OF
'   LINEAR EQUATIONS MAY BE SOLVED.
'
'   USAGE
'   CALL DLLSQ (A,B,M,N,L,X,IPIV,EPS,IER,AUX)
'
'   DESCRIPTION OF PARAMETERS
'    A - DOUBLE PRECISION M BY N COEFFICIENT MATRIX (not DESTROYED).
'    B - DOUBLE PRECISION M BY L RIGHT HAND SIDE MATRIX (not DESTROYED).
'    M - ROW NUMBER OF MATRICES A AND B.
'    N - COLUMN NUMBER OF MATRIX A, ROW NUMBER OF MATRIX X.
'    L - COLUMN NUMBER OF MATRICES B AND X.
'    X - DOUBLE PRECISION N BY L SOLUTION MATRIX.
'    IER - A RESULTING ERROR PARAMETER.
'    IPIV - INTEGER OUTPUT VECTOR OF DIMENSION N WHICH
'           CONTAINS INFORMATIONS ON COLUMN INTERCHANGES
'           IN MATRIX A. (SEE REMARK NO.3).
'    AUX - A DOUBLE PRECISION AUXILIARY STORAGE ARRAY OF
'          DIMENSION MAX(2*N,L). ON RETURN FIRST L LOCATIONS
'          OF AUX CONTAIN THE RESULTING LEAST SQUARES.
'    EPS - SINGLE PRECISION INPUT PARAMETER WHICH SPECIFIES
'          A RELATIVE TOLERANCE FOR DETERMINATION OF RANK OF
'          MATRIX A.
'
'   REMARKS
'    (1) NO ACTION BESIDES ERROR MESSAGE IER=-2 IN CASE
'        M LESS THAN N.
'    (2) NO ACTION BESIDES ERROR MESSAGE IER=-1 IN CASE
'        OF A ZERO-MATRIX A.
'    (3) IF RANK K OF MATRIX A IS FOUND TO BE LESS THAN N BUT
'        GREATER THAN 0, THE PROCEDURE RETURNS WITH ERROR CODE
'        IER=K INTO CALLING PROGRAM. THE LAST N-K ELEMENTS OF
'        VECTOR IPIV DENOTE THE USELESS COLUMNS IN MATRIX A.
'        THE REMAINING USEFUL COLUMNS FORM A BASE OF MATRIX A.
'    (4) IF THE PROCEDURE WAS SUCCESSFUL, ERROR PARAMETER IER
'        IS SET TO 0.
'
'   SUBROUTINES AND FUNCTION SUBPROGRAMS REQUIRED
'   NONE  (Per la versione VB: IF_ARI)
'
'   METHOD
'    HOUSEHOLDER TRANSFORMATIONS ARE USED TO TRANSFORM MATRIX A
'    TO UPPER TRIANGULAR FORM. AFTER HAVING APPLIED THE SAME
'    TRANSFORMATION TO THE RIGHT HAND SIDE MATRIX B, AN
'    APPROXIMATE SOLUTION OF THE PROBLEM IS COMPUTED BY
'    BACK SUBSTITUTION. FOR REFERENCE, SEE
'    G. GOLUB, NUMERICAL METHODS FOR SOLVING LINEAR LEAST
'    SQUARES PROBLEMS, NUMERISCHE MATHEMATIK, VOL.7,
'    ISS.3 (1965), PP.206-216.
'
' ..................................................................
'
    Dim I&, J&, K&, L&, M&, N&, II&, ID&, IST&, IEND&, JST&, KPIV&, KST&, LM&, LN&
    Dim BETA#, h#, PIV#, SIG#, Tol#
'
    M = UBound(a, 1)
    N = UBound(a, 2)
    L = UBound(B, 2)
'
    ' ERROR TEST
    On IF_ARI(M - N) GoTo 30, 1, 1
'
    ' Trasforma le matrici in ingresso
    ' nei vettori usati da questa routine:
1   ReDim A_L#(1 To M * N), B_L#(1 To M * L), X_L#(1 To N * L), AUX_L#(1 To 2 * N * L)
    K = 0
    For I = 1 To N
        For J = 1 To M
            K = K + 1
            A_L(K) = a(J, I)
        Next J
    Next I
'
    K = 0
    For I = 1 To L
        For J = 1 To M
            K = K + 1
            B_L(K) = B(J, I)
        Next J
    Next I
'
    ' GENERATION OF INITIAL VECTOR S(K) (K=1,2,...,N) IN STORAGE
    ' LOCATIONS AUX_L(K) (K=1,2,...,N)
    PIV = 0#
    IEND = 0
    For K = 1 To N
        IPIV(K) = K
        h = 0#
        IST = IEND + 1
        IEND = IEND + M
        For I = IST To IEND
2           h = h + A_L(I) * A_L(I)
        Next I
        AUX_L(K) = h
        On IF_ARI(h - PIV) GoTo 4, 4, 3
3       PIV = h
        KPIV = K
4       'CONTINUE
    Next K
'
    ' ERROR TEST
    On IF_ARI(PIV) GoTo 31, 31, 5
'
    ' DEFINE TOLERANCE FOR CHECKING RANK OF A
5   SIG = Sqr(PIV)
    Tol = SIG * Abs(Eps)
'
'
    ' DECOMPOSITION LOOP
    LM = L * M
    IST = -M
    For K = 1 To N
        IST = IST + M + 1
        IEND = IST + M - K
        I = KPIV - K
        On IF_ARI(I) GoTo 8, 8, 6
'
        ' INTERCHANGE K-TH COLUMN OF A WITH KPIV-TH IN CASE KPIV.GT.K
6       h = AUX_L(K)
        AUX_L(K) = AUX_L(KPIV)
        AUX_L(KPIV) = h
        ID = I * M
        For I = IST To IEND
            J = I + ID
            h = A_L(I)
            A_L(I) = A_L(J)
7           A_L(J) = h
        Next I
'
        ' COMPUTATION OF PARAMETER SIG
8       On IF_ARI(K - 1) GoTo 11, 11, 9
9       SIG = 0#
        For I = IST To IEND
10          SIG = SIG + A_L(I) * A_L(I)
        Next I
        SIG = Sqr(SIG)
'
        ' TEST ON SINGULARITY
        On IF_ARI(SIG - Tol) GoTo 32, 32, 11
'
        ' GENERATE CORRECT SIGN OF PARAMETER SIG
11      h = A_L(IST)
        On IF_ARI(h) GoTo 12, 13, 13
12      SIG = -SIG
'
        ' SAVE INTERCHANGE INFORMATION
13      IPIV(KPIV) = IPIV(K)
        IPIV(K) = KPIV
'
        ' GENERATION OF VECTOR UK IN K-TH COLUMN OF MATRIX A AND OF
        ' PARAMETER BETA
        BETA = h + SIG
        A_L(IST) = BETA
        BETA = 1# / (SIG * BETA)
        J = N + K
        AUX_L(J) = -SIG
        On IF_ARI(K - N) GoTo 14, 19, 19
'
        ' TRANSFORMATION OF MATRIX A
14      PIV = 0#
        ID = 0
        JST = K + 1
        KPIV = JST
        For J = JST To N
            ID = ID + M
            h = 0#
            For I = IST To IEND
                II = I + ID
15              h = h + A_L(I) * A_L(II)
            Next I
            h = BETA * h
            For I = IST To IEND
                II = I + ID
16              A_L(II) = A_L(II) - A_L(I) * h
            Next I
'
            ' UPDATING OF ELEMENT S(J) STORED IN LOCATION AUX_L(J)
            II = IST + ID
            h = AUX_L(J) - A_L(II) * A_L(II)
            AUX_L(J) = h
            On IF_ARI(h - PIV) GoTo 18, 18, 17
17          PIV = h
            KPIV = J
18          'CONTINUE
        Next J
'
        ' TRANSFORMATION OF RIGHT HAND SIDE MATRIX B
19      For J = K To LM Step M
            h = 0#
            IEND = J + M - K
            II = IST
            For I = J To IEND
                h = h + A_L(II) * B_L(I)
20              II = II + 1
            Next I
            h = BETA * h
            II = IST
            For I = J To IEND
                B_L(I) = B_L(I) - A_L(II) * h
21              II = II + 1
            Next I
        Next J
    Next K
    ' END OF DECOMPOSITION LOOP
'
'
    ' BACK SUBSTITUTION AND BACK INTERCHANGE
    IER = 0
    I = N
    LN = L * N
    PIV = 1# / AUX_L(2 * N)
    For K = N To LN Step N
        X_L(K) = PIV * B_L(I)
22      I = I + M
    Next K
    On IF_ARI(N - 1) GoTo 26, 26, 23
23  JST = (N - 1) * M + N
    For J = 2 To N
        JST = JST - M - 1
        K = N + N + 1 - J
        PIV = 1# / AUX_L(K)
        KST = K - N
        ID = IPIV(KST) - KST
        IST = 2 - J
        For K = 1 To L
            h = B_L(KST)
            IST = IST + N
            IEND = IST + J - 2
            II = JST
            For I = IST To IEND
                II = II + M
24              h = h - A_L(II) * X_L(I)
            Next I
            I = IST - 1
            II = I + ID
            X_L(I) = X_L(II)
            X_L(II) = PIV * h
25          KST = KST + M
        Next K
    Next J
'
'
    ' COMPUTATION OF LEAST SQUARES
26  IST = N + 1
    IEND = 0
    For J = 1 To L
        IEND = IEND + M
        h = 0#
        On IF_ARI(M - N) GoTo 29, 29, 27
27      For I = IST To IEND
28          h = h + B_L(I) * B_L(I)
        Next I
        IST = IST + M
29      AUX_L(J) = h
    Next J
'
    ' Riordina i valori dei vettori usati
    ' da questa routine nelle matrici dei
    ' risultati:
    K = 0
    For I = 1 To L
        For J = 1 To N
            K = K + 1
            X(J, I) = X_L(K)
        Next J
    Next I
'
    K = 0
    For I = 1 To L
            K = K + 1
            AUX(1, I) = AUX_L(K)
    Next I
    Exit Sub
'
    ' ERROR RETURN IN CASE M LESS THAN N
30  IER = -2
    Exit Sub
'
    ' ERROR RETURN IN CASE OF ZERO-MATRIX A
31  IER = -1
    Exit Sub
'
    ' ERROR RETURN IN CASE OF RANK OF MATRIX A LESS THAN N
32  IER = K - 1
    Exit Sub
'
'
'
End Sub
Private Function IF_ARI(ByVal vX As Variant) As Long
'
'   Da usare come:      On IF_ARI(vX) GoTo Ln1, Ln2, Ln3
'   Per implementare:   If (vX) Ln1, Ln2, Ln3
'                       che e' l' IF aritmetico del FORTRAN.
'
'   ATTENZIONE: nel programma chiamante NON USARE il :
'   dopo i numeri di linea, altrimenti si ottiene un
'   errore fatale "Unreferenced Memory Call at ...".
'
    If (vX < 0) Then
        IF_ARI = 1
    ElseIf (vX = 0) Then
        IF_ARI = 2
    Else
        IF_ARI = 3
    End If
'
'
'
End Function
Private Sub Balanc(ByRef a() As Double)
'
'   Given an N by N matrix a() stored in an array, this routine replaces
'   it by a balanced matrix with identical eigenvalues.
'   A symmetric matrix is already balanced and is unaffected
'   by this procedure. The parameter RADIX should be the machine's
'   floating-point radix.
'
    Dim I&, J&, N&, Last&
    Dim C#, f#, g#, r#, s#
    Const RADIX# = 2#, SQRDX# = RADIX * RADIX
'
    N = UBound(a, 1)
'
1
    Last = 1
    For I = 1 To N  ' Calculate row and column norms.
        C = 0#
        r = 0#
        For J = 1 To N
            If (J <> I) Then
                C = C + Abs(a(J, I))
                r = r + Abs(a(I, J))
            End If
        Next J
        If (C <> 0#) And (r <> 0#) Then ' If both are nonzero,
            g = r / RADIX
            f = 1#
            s = C + r
'
2
            If (C < g) Then ' find the integer power of the machine radix that
                            ' comes closest to balancing the matrix.
                f = f * RADIX
                C = C * SQRDX
                GoTo 2
            End If
            g = r * RADIX
'
3
            If (C > g) Then
                f = f / RADIX
                C = C / SQRDX
                GoTo 3
            End If
'
            If ((C + r) / f < 0.95 * s) Then
                Last = 0
                g = 1# / f
                For J = 1 To N ' Apply similarity transformation.
                    a(I, J) = a(I, J) * g
                Next J
                For J = 1 To N
                    a(J, I) = a(J, I) * f
                Next J
            End If
        End If
    Next I
    If (Last = 0) Then GoTo 1
'
'
'
End Sub
Public Sub TRISOL(a() As Double, ByRef B() As Double, C() As Double, ByRef d() As Double)
'
'   Risolve un sistema di equazioni tridiagonale
'   con il metodo di Thomas.
'    a(1 To N): diagonale bassa.
'    b(1 To N): diagonale centrale.
'    c(1 To N): diagonale alta.
'    d(1 To N): entra con il vettore dei termini noti;
'               esce con il vettore delle soluzioni.
'   Deve essere:
'    a(1) = 0.
'    c(N) = 0.
'
'   Da: Finite Difference Methods in Heat Transfer
'       M. Necati Ozisik
'       CRC Press
'
    Dim I&, J&, N&, r#
'
    N = UBound(a)
'
    For I = 2 To N
        r = a(I) / B(I - 1)
        B(I) = B(I) - r * C(I - 1)
        d(I) = d(I) - r * d(I - 1)
    Next I
'
    d(N) = d(N) / B(N)
    For I = 2 To N
        J = N - I + 1
        d(J) = (d(J) - C(J) * d(J + 1)) / B(J)
    Next I
'
'
'
End Sub
Public Function Jacobi(ByRef a() As Double, ByRef d() As Double, ByRef v() As Double, _
    Optional ByRef NRot As Long) As Boolean
'
'   Computes all eigenvalues and eigenvectors of a real symmetric matrix a(),
'   which is of size N by N. Only the elements of a() above the diagonal are cosidered.
'   On output, elements of a() above the diagonal are destroyed.
'   d() returns the eigenvalues of a() in its first N elements.
'   v() is a matrix with the same logical and physical dimensions as a(),
'   whose columns contain, on output, the normalized eigenvectors of a().
'   NRot returns the number of Jacobi rotations that were required.
'
    Dim I&, IP&, IQ&, J&, N&
    Dim C#, g#, h#, s#, SM#, t#, tau#, Theta#, tresh#
'
    N = UBound(a, 1)
    ReDim d(1 To N), v(1 To N, 1 To N)
    ReDim B#(1 To N), z#(1 To N)
'
    For IP = 1 To N         ' Initialize to the identity matrix.
        For IQ = 1 To N
            v(IP, IQ) = 0#
        Next IQ
        v(IP, IP) = 1#
    Next IP
'
    For IP = 1 To N
        B(IP) = a(IP, IP)   ' Initialize b and d to the diagonal of a.
        d(IP) = B(IP)
        z(IP) = 0#          ' This vector will accumulate terms of the form TApq
                            ' as in equation (11.1.14).
    Next IP
'
    NRot = 0
    For I = 1 To 50
        SM = 0#
        For IP = 1 To N - 1 ' Sum off-diagonal elements.
            For IQ = IP + 1 To N
                SM = SM + Abs(a(IP, IQ))
            Next IQ
        Next IP
        'If (sm = 0#) Then
        If (Abs(SM) < 1E-20) Then
            Jacobi = True   ' The normal return, which relies on
            Exit Function   ' quadratic convergence to machine underflow.
        End If
'
        If (I < 4) Then
            tresh = 0.2 * SM / (N ^ 2)  ' ...on the first three sweeps.
        Else
            tresh = 0#                  ' ...thereafter.
        End If
'
        For IP = 1 To N - 1
            For IQ = IP + 1 To N
                g = 100# * Abs(a(IP, IQ))
                ' After four sweeps, skip the rotation if the
                ' off-diagonal element is small:
                If ((I > 4) And ((Abs(d(IP)) + g) = Abs(d(IP))) _
                And ((Abs(d(IQ)) + g) = Abs(d(IQ)))) Then
                    a(IP, IQ) = 0#
                ElseIf (Abs(a(IP, IQ)) > tresh) Then
                    h = d(IQ) - d(IP)
                    If (Abs(h) + g = Abs(h)) Then
                        t = a(IP, IQ) / h
                    Else
                        Theta = 0.5 * h / a(IP, IQ) ' Equation (11.1.10).
                        t = 1# / (Abs(Theta) + Sqr(1# + Theta ^ 2))
                        If (Theta < 0#) Then t = -t
                    End If
                    C = 1# / Sqr(1 + t ^ 2)
                    s = t * C
                    tau = s / (1# + C)
                    h = t * a(IP, IQ)
                    z(IP) = z(IP) - h
                    z(IQ) = z(IQ) + h
                    d(IP) = d(IP) - h
                    d(IQ) = d(IQ) + h
                    a(IP, IQ) = 0#
                    For J = 1 To IP - 1         ' Case of rotations 1 <= j < p.
                        g = a(J, IP)
                        h = a(J, IQ)
                        a(J, IP) = g - s * (h + g * tau)
                        a(J, IQ) = h + s * (g - h * tau)
                    Next J
                    For J = IP + 1 To IQ - 1    ' Case of rotations p < j < q.
                        g = a(IP, J)
                        h = a(J, IQ)
                        a(IP, J) = g - s * (h + g * tau)
                        a(J, IQ) = h + s * (g - h * tau)
                    Next J
                    For J = IQ + 1 To N         ' Case of rotations q < j <= n.
                        g = a(IP, J)
                        h = a(IQ, J)
                        a(IP, J) = g - s * (h + g * tau)
                        a(IQ, J) = h + s * (g - h * tau)
                    Next J
                    For J = 1 To N
                        g = v(J, IP)
                        h = v(J, IQ)
                        v(J, IP) = g - s * (h + g * tau)
                        v(J, IQ) = h + s * (g - h * tau)
                    Next J
                    NRot = NRot + 1
                End If
            Next IQ
        Next IP
'
        For IP = 1 To N
            B(IP) = B(IP) + z(IP)
            d(IP) = B(IP)   ' Update d with the sum of tapq,
            z(IP) = 0#      ' and reinitialize z.
        Next IP
    Next I
'
'
Jacobi_ERR:
    Jacobi = False
    MsgBox "Troppe iterazioni.", vbCritical, " Jacobi"
'
'
'
End Function
Private Function ludcmp(ByRef a() As Double, ByRef Indx() As Long, _
    ByRef d As Double) As Boolean
'
'   Given a matrix a(1 To N, 1 To N), this routine replaces it by
'   the LU decomposition of a rowwise permutation of itself.
'   a() is the input matrix. a() is output, arranged as in equation
'   (2.3.14) above; Indx(1 To N) is an output vector that records the
'   row permutation effected by the partial pivoting;
'   d is output as  1 depending on whether the number of row interchanges
'   was even or odd, respectively. This routine is used in combination
'   with lubksb to solve linear equations or invert a matrix.
'
    Dim I&, IMax&, N&, J&, K&, bSing As Boolean
    Dim aamax#, dum#, sum#, VV#()
    Const tiny# = 1E-20     ' A small number.
'
    N = UBound(a, 1)
    ReDim Indx(1 To N)
    ReDim VV#(1 To N)       ' vv stores the implicit scaling of each row.
    bSing = False
'
    d = 1#                  ' No row interchanges yet.
    For I = 1 To N          ' Loop over rows to get the implicit scaling information.
        aamax = 0#
        For J = 1 To N
            If (Abs(a(I, J)) > aamax) Then aamax = Abs(a(I, J))
        Next J
        If (aamax = 0#) Then
            MsgBox "Singular matrix in ludcmp." & vbNewLine _
                 & "No nonzero largest element.", vbCritical, " ludcmp"
            ludcmp = False
            Exit Function
        End If
        VV(I) = 1# / aamax  ' Save the scaling.
    Next I
'
    For J = 1 To N          ' This is the loop over columns of Crout's method.
        For I = 1 To J - 1  ' This is equation (2.3.12) except for i = j.
            sum = a(I, J)
            For K = 1 To I - 1
                sum = sum - a(I, K) * a(K, J)
            Next K
            a(I, J) = sum
        Next I
        aamax = 0#          ' Initialize for the search for largest pivot element.
        For I = J To N      ' This is i = j of equation (2.3.12) and i = j +1 ... N
                            ' of equation (2.3.13).
            sum = a(I, J)
            For K = 1 To J - 1
                sum = sum - a(I, K) * a(K, J)
            Next K
            a(I, J) = sum
            dum = VV(I) * Abs(sum)  ' Figure of merit for the pivot.
            If (dum >= aamax) Then  ' Is it better than the best so far?
                IMax = I
                aamax = dum
            End If
        Next I
'
        If (J <> IMax) Then         ' Do we need to interchange rows?
            For K = 1 To N          ' Yes, do so...
                dum = a(IMax, K)
                a(IMax, K) = a(J, K)
                a(J, K) = dum
            Next K
            d = -d                  ' ...and change the parity of d.
            VV(IMax) = VV(J)        ' Also interchange the scale factor.
        End If
        Indx(J) = IMax
1       If (a(J, J) = 0#) Then      ' If the pivot element is zero the matrix is singular
            a(J, J) = tiny          ' (at least to the precision of the algorithm).
                                    ' For some applications on singular matrices, it is
                                    ' desirable to substitute TINY for zero.
            bSing = True
        End If
'
        If (J <> N) Then            ' Now, finally, divide by the pivot element.
            dum = 1# / a(J, J)
            For I = J + 1 To N
                a(I, J) = a(I, J) * dum
            Next I
        End If
    Next J                          ' Go back for the next column in the reduction.
'
    ludcmp = True
    ludcmp = ludcmp And (Not bSing) ' Opzionale: per avvertire della singolarita'.
                                    ' Vedi commento originale all' istruzione 1
                                    ' e sg.
'
'
'
End Function
Private Function Cludcmp(ByRef Ca() As Complex, ByRef Indx&(), ByRef d#) As Boolean
'
'   Given a matrix Ca(1 To N, 1 To N), this routine replaces it by
'   the LU decomposition of a rowwise permutation of itself.
'   Ca() is the input matrix. Ca() is output, arranged as in equation
'   (2.3.14) above; Indx(1 To N) is an output vector that records the
'   row permutation effected by the partial pivoting;
'   d is output as  1 depending on whether the number of row interchanges
'   was even or odd, respectively. This routine is used in combination
'   with lubksb to solve linear equations or invert a matrix.
'
'   Versione per matrici a valori complessi.
'
    Dim I&, IMax&, N&, J&, K&, bSing As Boolean
    Dim aamax#, dum#, VV#(), Csum As Complex, Cdum As Complex
    Const tiny# = 1E-20     ' A small number.
'
    N = UBound(Ca, 1)
    ReDim Indx(1 To N)
    ReDim VV#(1 To N)       ' vv stores the implicit scaling of each row.
    bSing = False
'
    d = 1#                  ' No row interchanges yet.
    For I = 1 To N          ' Loop over rows to get the implicit scaling information.
        aamax = 0#
        For J = 1 To N
            If (CAbs(Ca(I, J)) > aamax) Then aamax = CAbs(Ca(I, J))
        Next J
        If (aamax = 0#) Then
            MsgBox "Singular matrix in ludcmp." & vbNewLine _
                 & "No nonzero largest element.", vbCritical, " ludcmp"
            Cludcmp = False
            Exit Function
        End If
        VV(I) = 1# / aamax  ' Save the scaling.
    Next I
'
    For J = 1 To N          ' This is the loop over columns of Crout's method.
        For I = 1 To J - 1  ' This is equation (2.3.12) except for i = j.
            Csum = Ca(I, J)
            For K = 1 To I - 1
                Csum = CDif(Csum, CMol(Ca(I, K), Ca(K, J)))
            Next K
            Ca(I, J) = Csum
        Next I
'
        aamax = 0#          ' Initialize for the search for largest pivot element.
        For I = J To N      ' This is i = j of equation (2.3.12) and i = j +1 ... N
                            ' of equation (2.3.13).
            Csum = Ca(I, J)
            For K = 1 To J - 1
                Csum = CDif(Csum, CMol(Ca(I, K), Ca(K, J)))
            Next K
            Ca(I, J) = Csum
            dum = VV(I) * CAbs(Csum)  ' Figure of merit for the pivot.
            If (dum >= aamax) Then  ' Is it better than the best so far?
                IMax = I
                aamax = dum
            End If
        Next I
'
        If (J <> IMax) Then         ' Do we need to interchange rows?
            For K = 1 To N          ' Yes, do so...
                Cdum = Ca(IMax, K)
                Ca(IMax, K) = Ca(J, K)
                Ca(J, K) = Cdum
            Next K
            d = -d                  ' ...and change the parity of d.
            VV(IMax) = VV(J)        ' Also interchange the scale factor.
        End If
        Indx(J) = IMax
1       If CEgu(Ca(J, J), CCmp(0#, 0#)) Then ' If the pivot element is zero the matrix is singular
            Ca(J, J) = CCmp(tiny, tiny)      ' (at least to the precision of the algorithm).
                                             ' For some applications on singular matrices, it is
                                             ' desirable to substitute TINY for zero.
            bSing = True
        End If
'
        If (J <> N) Then            ' Now, finally, divide by the pivot element.
            Cdum = CDiv(CCmp(1#, 0#), Ca(J, J))
            For I = J + 1 To N
                Ca(I, J) = CMol(Ca(I, J), Cdum)
            Next I
        End If
    Next J                          ' Go back for the next column in the reduction.
'
    Cludcmp = True
    Cludcmp = Cludcmp And (Not bSing)   ' Opzionale: per avvertire della singolarita'.
                                        ' Vedi commento originale all' istruzione 1
                                        ' e sg.
'
'
'
End Function
Private Sub lubksb(a() As Double, Indx() As Long, ByRef B() As Double)
'
'   Solves the set of N linear equations A * X = B.
'   Here a(1 To N, 1 To N) is input, not as the matrix A but rather
'   as its LU decomposition, determined by the routine ludcmp.
'   Indx(1 To N) is input as the permutation vector returned by ludcmp.
'   b(1 To N) is input as the right-hand side vector B, and returns
'   with the solution vector X.
'   a() and Indx() are not modified by this routine and can be left
'   in place for successive calls with different right-hand sides b().
'   This routine takes into account the possibility that b() will begin
'   with many zero elements, so it is efficient for use in matrix inversion.
'
    Dim I&, N&, II&, J&, LL&
    Dim sum#
'
    N = UBound(a, 1)
'
    II = 0              ' When II is set to a positive value, it will become the in-
                        ' dex of the first nonvanishing element of b(). We now do
                        ' the forward substitution, equation (2.3.6). The only new
                        ' wrinkle is to unscramble the permutation as we go.
    For I = 1 To N
        LL = Indx(I)
        sum = B(LL)
        B(LL) = B(I)
        If (II <> 0) Then
            For J = II To I - 1
                sum = sum - a(I, J) * B(J)
            Next J
        ElseIf (sum <> 0#) Then
            II = I      ' A nonzero element was encountered, so from now on we will
                        ' have to do the sums in the loop above.
        End If
        B(I) = sum
    Next I
'
    For I = N To 1 Step -1      ' Now we do the backsubstitution, equation (2.3.7).
        sum = B(I)
        For J = I + 1 To N
            sum = sum - a(I, J) * B(J)
        Next J
        B(I) = sum / a(I, I)    ' Store a component of the solution vector X.
    Next I
'
'
'
End Sub
Private Sub Clubksb(Ca() As Complex, Indx&(), ByRef Cb() As Complex)
'
'   Solves the set of N linear equations A * X = B.
'   Here Ca(1 To N, 1 To N) is input, not as the matrix A but rather
'   as its LU decomposition, determined by the routine ludcmp.
'   Indx(1 To N) is input as the permutation vector returned by ludcmp.
'   Cb(1 To N) is input as the right-hand side vector B, and returns
'   with the solution vector X.
'   Ca() and Indx() are not modified by this routine and can be left
'   in place for successive calls with different right-hand sides Cb().
'   This routine takes into account the possibility that Cb() will begin
'   with many zero elements, so it is efficient for use in matrix inversion.
'
'   Versione per matrici a valori complessi.
'
    Dim I&, N&, II&, J&, LL&
    Dim Csum As Complex
'
    N = UBound(Ca, 1)
'
    II = 0              ' When II is set to a positive value, it will become the in-
                        ' dex of the first nonvanishing element of Cb(). We now do
                        ' the forward substitution, equation (2.3.6). The only new
                        ' wrinkle is to unscramble the permutation as we go.
    For I = 1 To N
        LL = Indx(I)
        Csum = Cb(LL)
        Cb(LL) = Cb(I)
        If (II <> 0) Then
            For J = II To I - 1
                Csum = CDif(Csum, CMol(Ca(I, J), Cb(J)))
            Next J
        ElseIf (Not CEgu(Csum, CCmp(0#, 0#))) Then
            II = I      ' A nonzero element was encountered, so from now on we will
                        ' have to do the sums in the loop above.
        End If
        Cb(I) = Csum
    Next I
'
    For I = N To 1 Step -1      ' Now we do the backsubstitution, equation (2.3.7).
        Csum = Cb(I)
        For J = I + 1 To N
            Csum = CDif(Csum, CMol(Ca(I, J), Cb(J)))
        Next J
        Cb(I) = CDiv(Csum, Ca(I, I))    ' Store a component of the solution vector X.
    Next I
'
'
'
End Sub
Public Function Inversa_2(ByRef a() As Double, ByRef C() As Double _
    , Optional ByRef det As Double) As Boolean
'
'   Ritorna in c() l' inversa della matrice quadrata a()
'   ed in det il determinante di a().
'   La matrice originale a() viene modificata.
'   Se durante il calcolo vengono trovate condizioni
'   anomale la funzione ritorna False.
'   Nota: l' algoritmo usato non e' molto robusto:
'         per matrici molto grandi e' preferibile
'         usare la routine Inversa.
'
'   Da: Basic Computer Programs in Science and Engineering
'       J. H. Gilder
'       Hayden Book Company.
'
    Dim I As Long, J As Long, L As Long, K As Long, N As Long
    Dim t As Double, s As Double, d As Double
'
    On Error GoTo Inversa_2_ERR
'
    N = UBound(a, 1)
    ReDim C(1 To N, 1 To N) As Double
'
    For I = 1 To N
        For J = 1 To N
            C(I, J) = 0#
        Next J
        C(I, I) = 1#
    Next I
'
    det = 1#
    For J = 1 To N
        For I = J To N
            If (a(I, J) <> 0) Then Exit For
        Next I
        If (I > N) Then Err.Raise 5, , "La matrice e' singolare."
'
        For L = 1 To N
            t = a(J, L)
            a(J, L) = a(I, L)
            a(I, L) = t
            s = C(J, L)
            C(J, L) = C(I, L)
            C(I, L) = s
        Next L
'
        d = a(J, J)
        det = det * d
        For L = 1 To N
            a(J, L) = a(J, L) / d
            C(J, L) = C(J, L) / d
        Next L
'
        For L = 1 To N
            If (J <> L) Then
                d = a(L, J)
                For K = 1 To N
                    a(L, K) = a(L, K) - a(J, K) * d
                    C(L, K) = C(L, K) - C(J, K) * d
                Next K
            End If
        Next L
    Next J
'
'
Inversa_2_ERR:
    Inversa_2 = (Err.Number = 0)
    If (Err.Number <> 0) Then
        MsgBox Err.Description, vbCritical, " Inversa_2"
    End If
'
'
'
End Function
Public Function Inversa(ByRef a() As Double, ByRef C() As Double) As Boolean
'
'   Ritorna in c() l' inversa della matrice quadrata a().
'   La matrice originale a() viene sostituita dalla sua
'   scomposizione in matrici triangolari inferiore e superiore:
'
    Dim I&, J&, N&, Indx&(), d#
'
    N = UBound(a, 1)
    ReDim C(1 To N, 1 To N), B#(1 To N)
'
    For I = 1 To N              ' Set up identity matrix.
        For J = 1 To N
            C(I, J) = 0#
        Next J
        C(I, I) = 1#
    Next I
'
    ' Decompose the matrix just once:
    If ludcmp(a(), Indx(), d) Then
        ' Find inverse by columns:
        For J = 1 To N
            For I = 1 To N
                B(I) = C(I, J)
            Next I
            Call lubksb(a(), Indx(), B())
            For I = 1 To N
                C(I, J) = B(I)
            Next I
        Next J
    '
        Inversa = True
    Else
        Inversa = False
    End If
'
'
'
End Function
Public Function CInversa(ByRef Ca() As Complex, ByRef CC() As Complex) As Boolean
'
'   Ritorna in Cc(), a valori complessi, l' inversa della matrice quadrata
'   a valori complessi Ca().
'   La matrice originale Ca() viene sostituita dalla sua
'   scomposizione in matrici triangolari inferiore e superiore:
'
    Dim I&, J&, N&, Indx&(), d#
'
    N = UBound(Ca, 1)
    ReDim CC(1 To N, 1 To N) As Complex, Cb(1 To N) As Complex
'
    For I = 1 To N              ' Set up identity matrix.
        For J = 1 To N
            CC(I, J) = CCmp(0#, 0#)
        Next J
        CC(I, I) = CCmp(1#, 0#)
    Next I
'
    ' Decompose the matrix just once:
    If Cludcmp(Ca(), Indx(), d) Then
'        ' Find inverse by columns:
        For J = 1 To N
            For I = 1 To N
                Cb(I) = CC(I, J)
            Next I
            Call Clubksb(Ca(), Indx(), Cb())
            For I = 1 To N
                CC(I, J) = Cb(I)
            Next I
        Next J
'
        CInversa = True
    Else
        CInversa = False
    End If
'
'
'
End Function
Public Function Determinante(ByRef a() As Double, ByRef det As Double) As Boolean
'
'   Ritorna, in det, il determinante della matrice quadrata a().
'   La matrice originale a() viene sostituita dalla sua
'   scomposizione in matrici triangolari inferiore e superiore:
'
    Dim Indx&(), J&
'
    ' This returns det as  1:
    If ludcmp(a(), Indx(), det) Then
        For J = 1 To UBound(a, 1)
            det = det * a(J, J)
        Next J
    '
        Determinante = True
    Else
        Determinante = False
    End If
'
'
'
End Function
Public Function CDeterminante(ByRef Ca() As Complex, ByRef Cdet As Complex) As Boolean
'
'   Ritorna, in Cdet complesso, il determinante della matrice quadrata
'   a valori complessi Ca().
'   La matrice originale Ca() viene sostituita dalla sua
'   scomposizione in matrici triangolari inferiore e superiore:
'
    Dim Indx&(), J&, det#
'
    ' This returns det as  1:
    If Cludcmp(Ca(), Indx(), det) Then
        Cdet = CCmp(det, 0#)
        For J = 1 To UBound(Ca, 1)
            Cdet = CMol(Cdet, Ca(J, J))
        Next J
    '
        CDeterminante = True
    Else
        CDeterminante = False
    End If
'
'
'
End Function
Public Function LinEqu(ByRef a() As Double, ByRef B() As Double) As Boolean
'
'   Risolve un sistema di N equazioni lineari in N incognite.
'    a(1 To N, 1 To N): matrice dei coefficienti reali delle incognite.
'    b(1 To N):         vettore dei termini noti reali.
'   La matrice originale a() viene sostituita dalla sua
'   scomposizione in matrici triangolari inferiore e superiore.
'   Il vettore b() ritorna con le soluzioni:
'
    Dim Indx&(), d#
'
    If ludcmp(a(), Indx(), d) Then
        Call lubksb(a(), Indx(), B())
'
        LinEqu = True
    Else
        LinEqu = False
    End If
'
'
'
End Function
Public Function CLinEqu(ByRef Ca() As Complex, ByRef Cb() As Complex) As Boolean
'
'   Risolve un sistema di N equazioni lineari in N incognite.
'    Ca(1 To N, 1 To N): matrice dei coefficienti complessi delle incognite.
'    Cb(1 To N):         vettore dei termini noti complessi.
'   La matrice originale Ca() viene sostituita dalla sua
'   scomposizione in matrici triangolari inferiore e superiore.
'   Il vettore Cb() ritorna con le soluzioni complesse:
'
    Dim Indx&(), d#
'
    If Cludcmp(Ca(), Indx(), d) Then
        Call Clubksb(Ca(), Indx(), Cb())
'
        CLinEqu = True
    Else
        CLinEqu = False
    End If
'
'
'
End Function
Private Sub elmhes(ByRef a() As Double)
'
'   Reduction to Hessenberg form by the elimination method.
'   The real, nonsymmetric, N by N matrix a(), is replaced by an upper Hessenberg
'   matrix with identical eigenvalues.
'   Recommended, but not required, is that this routine be preceded by balanc.
'   On output, the Hessenberg matrix is in elements a(i,j) with i <= j + 1.
'   Elements with i > j + 1 are to be thought of as zero, but are returned with
'   random values.
'
    Dim I&, J&, M&, N&
    Dim X#, Y#
'
    N = UBound(a, 1)
'
    For M = 2 To N - 1          ' m is called r + 1 in the text.
        X = 0#
        I = M
        For J = M To N          ' Find the pivot.
            If (Abs(a(J, M - 1)) > Abs(X)) Then
                X = a(J, M - 1)
                I = J
            End If
        Next J
'
        If (I <> M) Then        ' Interchange rows and columns.
            For J = M - 1 To N
                Y = a(I, J)
                a(I, J) = a(M, J)
                a(M, J) = Y
            Next J
            For J = 1 To N
                Y = a(J, I)
                a(J, I) = a(J, M)
                a(J, M) = Y
            Next J
        End If
'
        If (X <> 0#) Then       ' Carry out the elimination.
            For I = M + 1 To N
                Y = a(I, M - 1)
                If (Y <> 0#) Then
                    Y = Y / X
                    a(I, M - 1) = Y
                    For J = M To N
                        a(I, J) = a(I, J) - Y * a(M, J)
                    Next J
                    For J = 1 To N
                        a(J, M) = a(J, M) + Y * a(J, I)
                    Next J
                End If
            Next I
        End If
    Next M
'
'
'
End Sub
Private Function Hqr(ByRef a() As Double, ByRef wr() As Double, ByRef wi() As Double) As Boolean
'
'   Finds all eigenvalues of an N by N upper Hessenberg matrix a().
'   On input a() can be exactly as output from elmhes (11.5);
'   on output it is destroyed.
'   The real and imaginary parts of the eigenvalues are returned in
'   wr() and wi(), respectively.
'
    Dim I&, Its&, J&, K&, L&, M&, N&, NN&
    Dim anorm#, p#, q#, r#, s#, t#, u#, v#, w#, X#, Y#, z#
'
    On Error GoTo hqr_ERR
'
    N = UBound(a, 1)
    ReDim wr(1 To N), wi(1 To N)
'
    anorm = 0#      ' Compute matrix norm for possible use in locating
    For I = 1 To N  ' single small subdiagonal element.
        For J = MAX(I - 1, 1) To N
            anorm = anorm + Abs(a(I, J))
        Next J
    Next I
'
    NN = N
    t = 0#              ' Gets changed only by an exceptional shift.
'
1   If (NN >= 0.1) Then ' Begin search for next eigenvalue.
        Its = 0
2       For L = NN To 2 Step -1     ' Begin iteration: look for single small sub-
                                    ' diagonal element.
            s = Abs(a(L - 1, L - 1)) + Abs(a(L, L))
            If (s = 0#) Then s = anorm
            If ((Abs(a(L, L - 1)) + s) = s) Then GoTo 3
        Next L
        L = 1
3       X = a(NN, NN)
        If (L = NN) Then            ' One root found.
            wr(NN) = X + t
            wi(NN) = 0#
            NN = NN - 1
        Else
            Y = a(NN - 1, NN - 1)
            w = a(NN, NN - 1) * a(NN - 1, NN)
            If (L = NN - 1) Then    ' Two roots found...
                p = 0.5 * (Y - X)
                q = p ^ 2 + w
                z = Sqr(Abs(q))
                X = X + t
                If (q >= 0#) Then   ' ...a real pair.
                    z = p + SIGN(z, p)
                    wr(NN) = X + z
                    wr(NN - 1) = wr(NN)
                    If (z <> 0#) Then wr(NN) = X - w / z
                    wi(NN) = 0#
                    wi(NN - 1) = 0#
                Else                ' ...a complex pair.
                    wr(NN) = X + p
                    wr(NN - 1) = wr(NN)
                    wi(NN) = z
                    wi(NN - 1) = -z
                End If
                NN = NN - 2
'
            Else                    ' No roots found. Continue iteration.
                If (Its = 30) Then Err.Raise 5, , "Troppe iterazioni"
'
                If (Its = 10) Or (Its = 20) Then    'Form exceptional shift.
                    t = t + X
                    For I = 1 To NN
                        a(I, I) = a(I, I) - X
                    Next I
                    s = Abs(a(NN, NN - 1)) + Abs(a(NN - 1, NN - 2))
                    X = 0.75 * s
                    Y = X
                    w = -0.4375 * s ^ 2
                End If
'
                Its = Its + 1
                For M = NN - 2 To L Step -1 ' Form shift and then look for 2 consecu-
                                            ' tive small subdiagonal elements.
                    z = a(M, M)
                    r = X - z
                    s = Y - z
                    p = (r * s - w) / a(M + 1, M) + a(M, M + 1) 'Equation (11.6.23).
                    q = a(M + 1, M + 1) - z - r - s
                    r = a(M + 2, M + 1)
                    s = Abs(p) + Abs(q) + Abs(r)    ' Scale to prevent over ow or under ow.
                    p = p / s
                    q = q / s
                    r = r / s
                    If (M = L) Then GoTo 4
                    u = Abs(a(M, M - 1)) * (Abs(q) + Abs(r))
                    v = Abs(p) * (Abs(a(M - 1, M - 1)) + Abs(z) + Abs(a(M + 1, M + 1)))
                    If (u + v = v) Then GoTo 4      ' Equation (11.6.26).
                Next M
'
4               For I = M + 2 To NN
                    a(I, I - 2) = 0#
                    If (I <> M + 2) Then a(I, I - 3) = 0#
                Next I
                For K = M To NN - 1                 ' Double QR step on rows l to NN
                                                    ' and columns m to NN.
                    If (K <> M) Then
                        p = a(K, K - 1)             ' Begin setup of Householder vector.
                        q = a(K + 1, K - 1)
                        r = 0#
                        If (K <> NN - 1) Then r = a(K + 2, K - 1)
                        X = Abs(p) + Abs(q) + Abs(r)
                        If (X <> 0#) Then
                            p = p / X               ' Scale to prevent over ow or under ow.
                            q = q / X
                            r = r / X
                        End If
                    End If
                    s = SIGN(Sqr(p ^ 2 + q ^ 2 + r ^ 2), p)
                    If (s <> 0#) Then
                        If (K = M) Then
                            If (L <> M) Then a(K, K - 1) = -a(K, K - 1)
                        Else
                            a(K, K - 1) = -s * X
                        End If
                        p = p + s                   ' Equations (11.6.24).
                        X = p / s
                        Y = q / s
                        z = r / s
                        q = q / p
                        r = r / p
                        For J = K To NN             ' Row modification.
                            p = a(K, J) + q * a(K + 1, J)
                            If (K <> NN - 1) Then
                                p = p + r * a(K + 2, J)
                                a(K + 2, J) = a(K + 2, J) - p * z
                            End If
                            a(K + 1, J) = a(K + 1, J) - p * Y
                            a(K, J) = a(K, J) - p * X
                        Next J
                        For I = L To MIN(NN, K + 3) ' Column modification.
                            p = X * a(I, K) + Y * a(I, K + 1)
                            If (K <> NN - 1) Then
                                p = p + z * a(I, K + 2)
                                a(I, K + 2) = a(I, K + 2) - p * r
                            End If
                            a(I, K + 1) = a(I, K + 1) - p * q
                            a(I, K) = a(I, K) - p
                        Next I
                    End If
                Next K
                GoTo 2 ' ...for next iteration on current eigenvalue.
            End If
        End If
        GoTo 1 ' ...for next eigenvalue.
    End If
'
'
hqr_ERR:
    Hqr = (Err.Number = 0)
    If (Err.Number <> 0) Then
        MsgBox Err.Description, vbCritical, " hqr"
    End If
'
'
'
End Function
Private Function MAX(ParamArray vD() As Variant) As Variant
'
'   Implementa la funzione MAX(D1, D2, ...DN) del FORTRAN:
'
    Dim J&, J1&, J2&, vDMax As Variant
'
    J1 = LBound(vD)
    J2 = UBound(vD)
    vDMax = vD(J1)
    For J = J1 + 1 To J2
        If (vDMax < vD(J)) Then vDMax = vD(J)
    Next J
'
    MAX = vDMax
'
'
'
End Function
Private Function MIN(ParamArray vD() As Variant) As Variant
'
'   Implementa la funzione MIN(D1, D2, ...DN) del FORTRAN:
'
    Dim J&, J1&, J2&, vDMin As Variant
'
    J1 = LBound(vD)
    J2 = UBound(vD)
    vDMin = vD(J1)
    For J = J1 + 1 To J2
        If (vD(J) < vDMin) Then vDMin = vD(J)
    Next J
'
    MIN = vDMin
'
'
'
End Function
Private Function SIGN(ByVal dV As Double, ByVal DS As Double) As Double
'
'   Ritorna il valore assoluto di v con il segno di s.
'   Implementa la funzione SIGN del FORTRAN:
'
    If (DS < 0) Then
        SIGN = -Abs(dV)
    Else
        SIGN = Abs(dV)
    End If
'
'
'
End Function
Public Function AutoValori(a() As Double, ByRef wr() As Double, ByRef wi() As Double) As Boolean
'
'   Calcola tutti gli autovalori, reali e complessi della
'   matrice a(), quadrata ed a valori reali.
'   La parte reale degli autovalori viene ritornata in wr()
'   ed in wi() quella complessa.
'   La funzione ritorna False se, durante i calcoli, vengono
'   trovate condizioni anomale.
'
    Balanc a()
    elmhes a()
'
    AutoValori = Hqr(a(), wr(), wi())
'
'
'
End Function
Public Function Trasposta(a() As Double) As Double()
'
'   Ritorna la trasposta della matrice a():
'
    Dim I&, J&
    ReDim C#(1 To UBound(a, 2), 1 To UBound(a, 1))
'
    For I = 1 To UBound(a, 1)
        For J = 1 To UBound(a, 2)
            C(J, I) = a(I, J)
        Next J
    Next I
'
    Trasposta = C()
'
'
'
End Function
Public Function CTrasposta(Ca() As Complex) As Complex()
'
'   Ritorna la trasposta della matrice a valori complessi Ca():
'
    Dim I&, J&
    ReDim CC(1 To UBound(Ca, 2), 1 To UBound(Ca, 1)) As Complex
'
    For I = 1 To UBound(Ca, 1)
        For J = 1 To UBound(Ca, 2)
            CC(J, I) = Ca(I, J)
        Next J
    Next I
'
    CTrasposta = CC()
'
'
'
End Function
Public Function CMConiugata(Ca() As Complex) As Complex()
'
'   Ritorna la coniugata della matrice a valori complessi Ca():
'
    Dim I&, J&
    Dim CC() As Complex
'
    CC() = Ca()
    
    For I = 1 To UBound(CC, 1)
        For J = 1 To UBound(CC, 2)
            CC(I, J).Im = -CC(I, J).Im
        Next J
    Next I
'
    CMConiugata = CC()
'
'
'
End Function
Public Function ReCMScMol(Ca() As Complex, Optional ByVal s As Double = 1) As Double()
'
'   Ritorna la parte reale della matrice a valori complessi Ca()
'   moltiplicata per lo scalare s:
'
    Dim I&, J&
    Dim C#()
'
    ReDim C(1 To UBound(Ca, 1), 1 To UBound(Ca, 2))
    
    For I = 1 To UBound(C, 1)
        For J = 1 To UBound(C, 2)
            C(I, J) = s * Ca(I, J).Re
        Next J
    Next I
'
    ReCMScMol = C()
'
'
'
End Function
Public Function ReCVScMol(Ca() As Complex, Optional ByVal s As Double = 1) As Double()
'
'   Ritorna la parte reale del vettore a valori complessi Ca()
'   moltiplicata per lo scalare s:
'
    Dim I&, J&
    Dim C#()
'
    ReDim C(1 To UBound(Ca, 1))
    
    For I = 1 To UBound(C)
        C(I) = s * Ca(I).Re
    Next I
'
    ReCVScMol = C()
'
'
'
End Function
Public Function VScMol(Ca() As Double, Optional ByVal s As Double = 1) As Double()
'
'   Ritorna il vettore a valori reali Ca()
'   moltiplicato per lo scalare s:
'
    Dim I&, J&
    Dim C#()
'
    ReDim C(1 To UBound(Ca, 1))
    
    For I = 1 To UBound(C)
        C(I) = s * Ca(I)
    Next I
'
    VScMol = C()
'
'
'
End Function
Public Function VDif(a() As Double, B() As Double) As Double()
'
'   Ritorna la differenza dei due vettori a() + (-b()).
'   I due vettori devono essere dello stesso ordine:
'
    Dim I&, J&
    ReDim C#(1 To UBound(a))
'
    On Error GoTo VDif_ERR
'
    If (UBound(a) <> UBound(B)) Then Err.Raise 5
'
    For I = 1 To UBound(a)
        C(I) = a(I) - B(I)
    Next I
'
    VDif = C()
'
'
VDif_ERR:
    If (Err.Number <> 0) Then
        MsgBox "I vettori non sono dello stesso ordine.", vbCritical, " VDif"
    End If
'
'
'
End Function
Public Function MDif(a() As Double, B() As Double) As Double()
'
'   Ritorna la differenza delle due matrici a() + (-b()).
'   Le due matrici devono essere dello stesso ordine:
'
    Dim I&, J&
    ReDim C#(1 To UBound(a, 1), 1 To UBound(a, 2))
'
    On Error GoTo MDif_ERR
'
    If (UBound(a, 1) <> UBound(B, 1)) Or (UBound(a, 2) <> UBound(B, 2)) Then Err.Raise 5
'
    For I = 1 To UBound(a, 1)
        For J = 1 To UBound(a, 2)
            C(I, J) = a(I, J) - B(I, J)
        Next J
    Next I
'
    MDif = C()
'
'
MDif_ERR:
    If (Err.Number <> 0) Then
        MsgBox "Le matrici non sono dello stesso ordine.", vbCritical, " MDif"
    End If
'
'
'
End Function
Public Function CMDif(Ca() As Complex, Cb() As Complex) As Complex()
'
'   Ritorna la differenza delle due matrici a valori complessi Ca() + (-Cb()).
'   Le due matrici devono essere dello stesso ordine:
'
    Dim I&, J&
    ReDim C(1 To UBound(Ca, 1), 1 To UBound(Ca, 2)) As Complex
'
    On Error GoTo CMDif_ERR
'
    If (UBound(Ca, 1) <> UBound(Cb, 1)) _
    Or (UBound(Ca, 2) <> UBound(Cb, 2)) Then Err.Raise 5
'
    For I = 1 To UBound(Ca, 1)
        For J = 1 To UBound(Ca, 2)
            C(I, J) = CDif(Ca(I, J), Cb(I, J))
        Next J
    Next I
'
    CMDif = C()
'
'
CMDif_ERR:
    If (Err.Number <> 0) Then
        MsgBox "Le matrici non sono dello stesso ordine.", vbCritical, " CMDif"
    End If
'
'
'
End Function
Public Function MMSc(a() As Double, ByVal s As Double) As Double()
'
'   Ritorna il prodotto della matrice a() per
'   lo scalare s:
'
    Dim I&, J&
    ReDim C#(1 To UBound(a, 1), 1 To UBound(a, 2))
'
    For I = 1 To UBound(a, 1)
        For J = 1 To UBound(a, 2)
            C(I, J) = s * a(I, J)
        Next J
    Next I
'
    MMSc = C()
'
'
'
End Function
Public Function MEgu(a() As Double, B() As Double) As Boolean
'
'   Verifica che le due matrici a() e b() siano uguali.
'   Le due matrici devono essere dello stesso ordine:
'
    Dim I&, J&
'
    On Error GoTo MEgu_ERR
'
    If (UBound(a, 1) <> UBound(B, 1)) Or (UBound(a, 2) <> UBound(B, 2)) Then Err.Raise 5
'
    For I = 1 To UBound(a, 1)
        For J = 1 To UBound(a, 2)
            If (a(I, J) <> B(I, J)) Then
                MEgu = False
                Exit Function
            End If
        Next J
    Next I
'
'
MEgu_ERR:
    MEgu = (Err.Number = 0)
    If (Err.Number <> 0) Then
        MsgBox "Le matrici non sono dello stesso ordine.", vbCritical, " MEgu"
    End If
'
'
'
End Function
Public Function CMEgu(Ca() As Complex, Cb() As Complex) As Boolean
'
'   Verifica che le due matrici a valori complessi Ca() e Cb() siano uguali.
'   Le due matrici devono essere dello stesso ordine:
'
    Dim I&, J&
'
    On Error GoTo CMEgu_ERR
'
    If (UBound(Ca, 1) <> UBound(Cb, 1)) _
    Or (UBound(Ca, 2) <> UBound(Cb, 2)) Then Err.Raise 5
'
    For I = 1 To UBound(Ca, 1)
        For J = 1 To UBound(Ca, 2)
            If (Not CEgu(Ca(I, J), Cb(I, J))) Then
                CMEgu = False
                Exit Function
            End If
        Next J
    Next I
'
'
CMEgu_ERR:
    CMEgu = (Err.Number = 0)
    If (Err.Number <> 0) Then
        MsgBox "Le matrici non sono dello stesso ordine.", vbCritical, " CMEgu"
    End If
'
'
'
End Function
Public Function MSom(a() As Double, B() As Double) As Double()
'
'   Ritorna la somma delle due matrici a() + b().
'   Le due matrici devono essere dello stesso ordine:
'
    Dim I&, J&
    ReDim C#(1 To UBound(a, 1), 1 To UBound(a, 2))
'
    On Error GoTo MSom_ERR
'
    If (UBound(a, 1) <> UBound(B, 1)) _
    Or (UBound(a, 2) <> UBound(B, 2)) Then Err.Raise 5
'
    For I = 1 To UBound(a, 1)
        For J = 1 To UBound(a, 2)
            C(I, J) = a(I, J) + B(I, J)
        Next J
    Next I
'
    MSom = C()
'
'
MSom_ERR:
    If (Err.Number <> 0) Then
        MsgBox "Le matrici non sono dello stesso ordine.", vbCritical, " MSom"
    End If
'
'
'
End Function
Public Function CMSom(Ca() As Complex, Cb() As Complex) As Complex()
'
'   Ritorna la somma delle due matrici a valori complessi Ca() + Cb().
'   Le due matrici devono essere dello stesso ordine:
'
    Dim I&, J&
    ReDim CC(1 To UBound(Ca, 1), 1 To UBound(Ca, 2)) As Complex
'
    On Error GoTo CMSom_ERR
'
    If (UBound(Ca, 1) <> UBound(Cb, 1)) _
    Or (UBound(Ca, 2) <> UBound(Cb, 2)) Then Err.Raise 5
'
    For I = 1 To UBound(Ca, 1)
        For J = 1 To UBound(Ca, 2)
            CC(I, J) = CSom(Ca(I, J), Cb(I, J))
        Next J
    Next I
'
    CMSom = CC()
'
'
CMSom_ERR:
    If (Err.Number <> 0) Then
        MsgBox "Le matrici non sono dello stesso ordine.", vbCritical, " CMSom"
    End If
'
'
'
End Function
Public Function MMol(a() As Double, B() As Double) As Double()
'
'   Ritorna il prodotto delle due matrici a() * b().
'   Il numero di colonne di a() deve essere uguale
'   al numero di righe di b():
'
    Dim I&, J&, K&
    ReDim C#(1 To UBound(a, 1), 1 To UBound(B, 2))
'
    On Error GoTo MMol_ERR
'
    If (UBound(a, 2) <> UBound(B, 1)) Then Err.Raise 5
'
    For I = 1 To UBound(a, 1)
        For J = 1 To UBound(B, 2)
            For K = 1 To UBound(a, 2)
                C(I, J) = C(I, J) + a(I, K) * B(K, J)
            Next K
        Next J
    Next I
'
    MMol = C()
'
'
MMol_ERR:
    If (Err.Number <> 0) Then
        MsgBox "Le matrici non hanno ordini compatibili.", vbCritical, " MMol"
    End If
'
'
'
End Function
Public Function CMMol(Ca() As Complex, Cb() As Complex) As Complex()
'
'   Ritorna il prodotto delle due matrici a valori complessi Ca() * Cb().
'   Il numero di colonne di Ca() deve essere uguale
'   al numero di righe di Cb():
'
    Dim I&, J&, K&
    ReDim C(1 To UBound(Ca, 1), 1 To UBound(Cb, 2)) As Complex
'
    On Error GoTo CMMol_ERR
'
    If (UBound(Ca, 2) <> UBound(Cb, 1)) Then Err.Raise 5
'
    For I = 1 To UBound(Ca, 1)
        For J = 1 To UBound(Cb, 2)
            For K = 1 To UBound(Ca, 2)
                C(I, J) = CSom(C(I, J), CMol(Ca(I, K), Cb(K, J)))
            Next K
        Next J
    Next I
'
    CMMol = C()
'
'
CMMol_ERR:
    If (Err.Number <> 0) Then
        MsgBox "Le matrici non hanno ordini compatibili.", vbCritical, " CMMol"
    End If
'
'
'
End Function
Public Function MIde(ByVal N As Long) As Double()
'
'   Ritorna una matrice identica (o matrice unita')
'   di ordine N con N >= 1:
'
    Dim I&
    ReDim C#(1 To N, 1 To N)
'
    For I = 1 To N
        C(I, I) = 1#
    Next I
'
    MIde = C()
'
'
'
End Function
Public Function CMIde(ByVal N&) As Complex()
'
'   Ritorna una matrice complessa identica (o matrice unita')
'   di ordine N con N >= 1:
'
    Dim I&
    ReDim CC(1 To N, 1 To N) As Complex
'
    For I = 1 To N
        CC(I, I) = CCmp(1#, 0#)
    Next I
'
    CMIde = CC()
'
'
'
End Function
Public Sub POWINV(a() As Double, ByRef X() As Double, _
    ByVal Alpha As Double, ByRef Lambda As Double, _
    Optional ByVal Tol As Double = 0.000001, Optional ByVal MITER As Long = 100, _
    Optional ByRef ITERM As Long)
'
'   Subroutine POWINV uses the inverse power method with scaling to
'   estimate Eigenvalues of a matrix A.  The calling program must supply
'   the matrix a(), an initial vector x() of euclidean length 1, a scalar
'   Alpha, a tolerance Tol and an integer MITER = maximum number of
'   iterations desired.  The subroutine returns when the difference of
'   two successive estimates is less than Tol in absolute value or when
'   MITER iteration have been executed.  In the first case, a flag, ITERM,
'   is set to 1 and in the second case ITERM is set to 2.  The approximate
'   Eigenvalue is returned as Lambda and an approximate Eigenvector as x().
'
'   Note di traduzione: il vettore x() viene dimensionato in questa routine
'                       e vengono assegnati i valori iniziali x(1, .., N) = 1/Sqr(N).
'
    Dim I&, J&, N&, ITR&, IERROR&
    Dim TEMP#, YScale#, Estold#, Estnew#
'
    N = UBound(a, 1)
    ReDim sa#(1 To N, 1 To N), sx#(1 To N), X#(1 To N), Y#(1 To N)
    For I = 1 To N
        X(I) = 1# / Sqr(CDbl(N))
    Next I
'
    ITR = 1
    For I = 1 To N
        For J = 1 To N
            sa(I, J) = a(I, J)
        Next J
    Next I
'
    ' Calculate the initial Eigenvalue approximation:
    For I = 1 To N
        sx(I) = X(I)
        a(I, I) = a(I, I) - Alpha
    Next I
'
    Call GAUS(a(), X(), Y(), IERROR)
    If (IERROR = 2) Then GoTo 10
'
    TEMP = 0#
    YScale = 0#
    For I = 1 To N
        TEMP = TEMP + Y(I) * sx(I)
        YScale = YScale + Y(I) * Y(I)
    Next I
    YScale = Sqr(YScale)
    Estold = 1# / TEMP + Alpha
'
    ' Inverse power method iteration with scaling:
4   ITR = ITR + 1
    For I = 1 To N
        For J = 1 To N
            a(I, J) = sa(I, J)
        Next J
    Next I
    For I = 1 To N
        X(I) = Y(I) / YScale
        sx(I) = X(I)
        a(I, I) = a(I, I) - Alpha
    Next I
'
    Call GAUS(a(), X(), Y(), IERROR)
    If (IERROR = 2) Then GoTo 10
'
    TEMP = 0#
    YScale = 0#
    For I = 1 To N
        TEMP = TEMP + Y(I) * sx(I)
        YScale = YScale + Y(I) * Y(I)
    Next I
    YScale = Sqr(YScale)
    Estnew = 1# / TEMP + Alpha
'
    ' Test for termination of the inverse power method iteration:
    If (Abs(Estnew - Estold) <= Tol) Then GoTo 8
    If (ITR >= MITER) Then GoTo 9
    Estold = Estnew
    GoTo 4
'
8   ITERM = 1
    Lambda = Estnew
    For I = 1 To N              ' Aggiunta al programma originale
        X(I) = Y(I) / YScale    ' per ottenere valori di scala
    Next I                      ' inferiore.
    Exit Sub
'
9   ITERM = 2
    Lambda = Estnew
    Exit Sub
'
    ' Set ITERM = 3 if the Gauss elimination step fails:
10  ITERM = 3
'
'
'
End Sub
Public Sub POWERM(a() As Double, ByRef X() As Double, ByRef Lambda As Double, _
    Optional ByVal Tol As Double = 0.000001, Optional ByVal MITER As Long = 100, _
    Optional ByRef ITERM As Long)
'
'   Subroutine POWERM uses the power method with scaling to estimate the
'   dominant Eigenvalue of a matrix A.  The calling program must supply
'   the matrix a(), an initial vector x() of euclidean length 1, a tolerance Tol
'   and an integer MITER = maximum number of power iterations desired.
'   The subroutine returns when the difference of two successive estimates is
'   less than Tol in absolute value or when MITER iteration have been executed.
'   In the first case, a flag, ITERM, is set to 1 and in the second case ITERM
'   is set to 2.  The approximate Eigenvalue is returned as Lambda and an
'   approximate Eigenvector as x().
'
'   Note di traduzione: il vettore x() viene dimensionato in questa routine
'                       e vengono assegnati i valori iniziali x(1, .., N) = 1/Sqr(N).
'
    Dim I&, J&, N&, ITR&, IERROR&
    Dim TEMP#, YScale#, Estold#, Estnew#
'
    N = UBound(a, 1)
    ReDim X#(1 To N), Y#(1 To N)
    For I = 1 To N
        X(I) = 1# / Sqr(CDbl(N))
    Next I
'
    ITR = 1
'
    ' Calculate the initial Eigenvalue approximation:
    For I = 1 To N
        Y(I) = 0#
        For J = 1 To N
            Y(I) = Y(I) + a(I, J) * X(J)
        Next J
    Next I
    TEMP = 0#
    YScale = 0#
    For I = 1 To N
        TEMP = TEMP + Y(I) * X(I)
        YScale = YScale + Y(I) * Y(I)
    Next I
    YScale = Sqr(YScale)
    Estold = TEMP
'
    ' Power method iteration with scaling:
3   ITR = ITR + 1
    For I = 1 To N
        X(I) = Y(I) / YScale
    Next I
    For I = 1 To N
        Y(I) = 0#
        For J = 1 To N
            Y(I) = Y(I) + a(I, J) * X(J)
        Next J
    Next I
    TEMP = 0#
    YScale = 0#
    For I = 1 To N
        TEMP = TEMP + Y(I) * X(I)
        YScale = YScale + Y(I) * Y(I)
    Next I
    YScale = Sqr(YScale)
    Estnew = TEMP
'
    ' Test for termination of the power method iteration:
    If (Abs(Estnew - Estold) <= Tol) Then GoTo 7
    If (ITR >= MITER) Then GoTo 8
    Estold = Estnew
    GoTo 3
'
7   ITERM = 1
    Lambda = Estnew
    Exit Sub
'
8   ITERM = 2
    Lambda = Estnew
'
'
'
End Sub
Public Sub GAUS(ByRef a() As Double, ByRef B() As Double, ByRef X() As Double, _
    ByRef IERROR As Long)
'
'   Subroutine GAUS uses Gauss elimination (without pivoting) to solve
'   the system A * X = B.  The calling program must supply the matrix A
'   and the vector B. Arrays a() and b() are destroyed in GAUS.
'   The solution is returned in x() and a flag, IERROR, is set to 1 if a()
'   is non singular and is set to 2 it a() is singular.
'
    Dim I&, J&, K&, N&, IP1&, NM1&, NMK&, NP1&
    Dim TEMP#, q#
'
    N = UBound(a, 1)
    ReDim X#(1 To N)
'
    NM1 = N - 1
    For I = 1 To NM1
        ' Search for non-zero pivot element and interchange rows if necessary.
        ' If non-zero pivot element is found, set IERROR = 2 and return.
        For J = I To N
            If (a(J, I) = 0#) Then GoTo 3
            For K = I To N
                TEMP = a(I, K)
                a(I, K) = a(J, K)
                a(J, K) = TEMP
            Next K
            TEMP = B(I)
            B(I) = B(J)
            B(J) = TEMP
            GoTo 4
3
        Next J
        GoTo 8
'
        ' Eliminate the coefficients of X(I) in rows I + 1 ... N:
4       IP1 = I + 1
        For K = IP1 To N
            q = -a(K, I) / a(I, I)
            a(K, I) = 0#
            B(K) = q * B(I) + B(K)
            For J = IP1 To N
                a(K, J) = q * a(I, J) + a(K, J)
            Next J
        Next K
    Next I
    If (a(N, N) = 0#) Then GoTo 8
'
    ' Backsolve the equivalent triangularized system, set IERROR = 1
    ' and return:
    X(N) = B(N) / a(N, N)
    NP1 = N + 1
    For K = 1 To NM1
        q = 0#
        NMK = N - K
        For J = 1 To K
            q = q + a(NMK, NP1 - J) * X(NP1 - J)
        Next J
        X(NMK) = (B(NMK) - q) / a(NMK, NMK)
    Next K
    IERROR = 1
    Exit Sub
'
8   IERROR = 2
'
'
'
End Sub
