Attribute VB_Name = "modMixerControl"
'=======================================================================================
' Descrizione.....: Collezione di routines per il controllo
'                   dei Mixers delle schede audio.
' Nome dei Files..: modMixerControl.bas
' Data............: 20/08/2003 - Marv Solowey.
' Adattamenti di..: F. Languasco  (Ver. 1.0)
' Aggiornamento...: 05/05/2004
' Aggiornamento...: 19/03/2006
' Aggiornamento...: 02/08/2006
' Nuova versione..: 08/09/2009 (Ver. 2.0)
' Aggiornamento...: 16/11/2009 (Ver. 2.1)
' Versione........: 2.1 a 32 Bits.
' Sistema.........: VB6 (SP5) sotto Windows XP (SP2).
' E-Mail..........: MC7061@mclink.it
' DownLoads a.....: http://www.flanguasco.org
'=======================================================================================
'
'========================================================================='
'   Queste routines derivano, dopo ampio rimaneggiamento (Ver. 2.0), da   '
'   quelle scritte originariamente da Marv Solowey, <mmsont@yahoo.ca>,    '
'   a cui vanno tutti i miei ringraziamenti per avermi offerto lo spunto  '
'   iniziale.                                                             '
'========================================================================='
'
'   Routines disponibili:
'---------------------------------------------------------------------------------------
'   Routine per cercare le schede audio (mixers) presenti su questo PC:
'   CercaSchedeAudio()  la lista delle schede audio disponibili viene
'                       composta nel vettore SchedeAudio$(1 To NSchedeAudio).
'---------------------------------------------------------------------------------------
'   Routine per cercare i Mixers con una destinazione di tipo SPEAKERS
'   presenti sulle schede audio di questo PC e le linee a loro associate:
'    CercaMixersSp()    la lista dei mixers disponibili viene
'                       composta nel vettore MixerSp(1 To NMixSp);
'                       la lista delle linee di "Playback" viene
'                       composta nel vettore MixerSp().SrcLine(0 To NSrcLines - 1).
'---------------------------------------------------------------------------------------
'   Routine per cercare i Mixers con una destinazione di tipo WAVEIN
'   presenti sulle schede audio di questo PC e le linee a loro associate:
'    CercaMixersWi()    la lista dei mixers disponibili viene
'                       composta nel vettore MixerWi(1 To NMixWi);
'                       la lista delle linee di "Recording" viene
'                       composta nel vettore MixerWi().SrcLine(0 To NSrcLines - 1).
'---------------------------------------------------------------------------------------
'   Routine per selezionare/deselezionare una "Source line";
'   devono, precedentemente, essere state chiamate le Functions
'   CercaMixersSp e/o CercaMixersWi:
'    ImpostaAbilitazioneLinea Mixer_I As Mixer_Type, [Ich As Long], _
'                             [bIsDestLine As Boolean]
'     Mixer_I:           mixer da utilizzare.
'     Ich:               indice della "Source line" da impostare sul mixer.
'     bIsDestLine:       se True imposta lo stato di una "Destination line";
'                        se False imposta lo stato di una "Source line";
'---------------------------------------------------------------------------------------
'   Routine per impostare il volume di una "Source line";
'   devono, precedentemente, essere state chiamate le Functions
'   CercaMixersSp e/o CercaMixersWi:
'    ImpostaVolumeLinea Mixer_I As Mixer_Type, Volume As Long, [Ich As Long], _
'                       [bIsDestLine As Boolean]
'     Mixer_I:           mixer da utilizzare.
'     Volume:            livello di volume da impostare (0 <= Volume <= 65535).
'     Ich:               indice della "Source line" da impostare sul mixer.
'     bIsDestLine:       se True imposta il volume di una "Destination line";
'                        se False imposta il volume di una "Source line";
'---------------------------------------------------------------------------------------
'   Routine per trovare i nomi delle "Source lines" attive nel mixer Mixer_I;
'   devono, precedentemente, essere state chiamate le Functions
'   CercaMixersSp e/o CercaMixersWi:
'    Function CercaNomiLineeAttive(Mixer_I As Mixer_Type [, I1 as Long]) As String
'     Function CercaNomiLineeAttive:  ritorna i nomi delle "Source lines" attive.
'     Mixer_I:           mixer da utilizzare.
'     I1:                indice della prima linea attiva.
'---------------------------------------------------------------------------------------
'   Routine per trovare le frequenze di campionamento ammesse da un
'   mixer in modalita' "Playback"; deve, precedentemente, essere stata
'   chiamata la Function CercaMixersSp:
'    Function CercaWaveOutFs(Mixer_I As Mixer_Type, FsList() As Long, _
                      [NCanali_V As Integer], [BitsPerSample_V As Integer]) As Long
'     Function CercaWaveOutFs:  ritorna il numero di frequenze di campionamento ammesse.
'     Mixer_I:           mixer da interrogare.
'     FsList():          lista delle di frequenze di campionamento ammesse.
'     NCanali_V:         mumero di canali desiderato (1 per Mono | 2 per Stereo).
'     BitsPerSample_V:   numero di bits per campione desiderato (8 | 16).
'---------------------------------------------------------------------------------------
'   Routine per trovare le frequenze di campionamento ammesse da un
'   mixer in modalita' "Recording"; deve, precedentemente, essere stata
'   chiamata la Function CercaMixersWi:
'    Function CercaWaveInFs(Mixer_I As Mixer_Type, FsList() As Long, _
                      [NCanali_V As Integer], [BitsPerSample_V As Integer]) As Long
'     Function CercaWaveInFs:  ritorna il numero di frequenze di campionamento ammesse.
'     Mixer_I:           mixer da interrogare.
'     FsList():          lista delle di frequenze di campionamento ammesse.
'     NCanali_V:         mumero di canali desiderato (1 per Mono | 2 per Stereo).
'     BitsPerSample_V:   numero di bits per campione desiderato (8 | 16).
'---------------------------------------------------------------------------------------
'   Routine per trovare la descrizione dell' errore lRet eventualmente generato
'   dalle "Waveform Functions" e dalle "Audio Mixer Functions":
'    Function WaveERR(lRet As Long) As String
'     Function WaveERR:  ritorna il numero e la descrizione dell' errore.
'     lRet:              numero dell' errore da descrivere.
'---------------------------------------------------------------------------------------
'
Option Explicit
Option Base 0
'
Public NSchedeAudio&        ' N di schede audio disponibili.
Public SchedeAudio$()       ' Vettore con i nomi delle schede audio disponibili.
'
' Proprieta' delle linee:
Private Type MixerLine_Type
    LineID As Long          ' Identificativo della linea.
    Name As String          ' Nome della linea.
    Controls As Long        ' Per usi futuri.
    Channels As Long        '  "   "    "
'
    MuteControlID As Long   ' Identificativo del controllo di tipo "Mute".
    bNotMute As Boolean     ' Se True la linea non e' silenziata.
    bMuteOK As Boolean      ' Se True il controllo "Mute" della linea esiste ed e'
                            ' modificabile.
'
    Volume As Long          ' Volume della linea; se Volume = -1 il controllo non e'
                            ' disponibile.
    VolumeStep As Long      ' Minima variazione possibile per il valore di Volume.
    VolumeName As String    ' Nome del controllo di volume.
'
    IrMUX As Long           ' Indice di corrispondenza [inversa] delle linee di tipo MUX.
End Type
'
' Proprieta' dei mixers:
Private Type Mixer_Type
    MixerID As Long             ' Identificativo del mixer.
    Name As String              ' Nome del mixer.
'
    SchedaAudioID As Long       ' Identificativo della scheda audio a cui il mixer
                                ' appartiene (da usare in in WaveOutOpen e WaveInOpen).
'
    NSrcLines As Long           ' N di "Source lines" collegate alla "Destination line".
    SrcLine() As MixerLine_Type ' Vettore delle proprieta' delle "Source lines".
'
    DstLine As MixerLine_Type   ' Proprieta' della "Destination line" del mixer:
                                ' solo linee di tipo SPEAKERS o WAVEIN.
'
    bDstIsMUX As Boolean        ' Se True il mixer ha la "Destination line" con un
                                ' controllo di tipo MUX.
    MUXControlID As Long        ' Identificativo del controllo di tipo MUX (se esiste).
    MUXControlItems As Long     ' N di controlli del MUX (linee collegate).
End Type
'
Public NMixSp&                  ' N di mixers disponibili con destinazione SPEAKERS.
Public MixerSp() As Mixer_Type  ' Vettore dei mixers disponibili con
                                ' una destinazione di tipo SPEAKERS.
'
Public NMixWi&                  ' N di mixers disponibili con destinazione WAVEIN.
Public MixerWi() As Mixer_Type  ' Vettore dei mixers disponibili con
                                ' una destinazione di tipo WAVEIN.
'
Private hmixer As Long          ' handler del mixer correntemente aperto.
'
'-- Costanti per API: ------------------------------------------------------------------
'
Private Const MAXPNAMELEN = 32     ' max product name length (including NULL)
'
Private Const MIXER_SHORT_NAME_CHARS = 16
Private Const MIXER_LONG_NAME_CHARS = 64
'
Private Const MIXER_GETLINEINFOF_DESTINATION = &H0&
Private Const MIXER_GETLINEINFOF_SOURCE = &H1&
'
Private Const MIXERLINE_LINEF_ACTIVE = &H1&
Private Const MIXERLINE_LINEF_DISCONNECTED = &H8000&
Private Const MIXERLINE_LINEF_SOURCE = &H80000000
'
Private Const MIXERLINE_COMPONENTTYPE_DST_FIRST = &H0&
Private Const MIXERLINE_COMPONENTTYPE_DST_SPEAKERS = (MIXERLINE_COMPONENTTYPE_DST_FIRST + 4)
Private Const MIXERLINE_COMPONENTTYPE_DST_WAVEIN = (MIXERLINE_COMPONENTTYPE_DST_FIRST + 7)
Private Const MIXER_GETLINEINFOF_COMPONENTTYPE = &H3&
'
Private Const MIXER_GETLINECONTROLSF_ALL = &H0&
Private Const MIXER_GETLINECONTROLSF_ONEBYTYPE = &H2&
'
' Mixer control types:
Private Const MIXERCONTROL_CT_CLASS_SWITCH = &H20000000
Private Const MIXERCONTROL_CT_CLASS_FADER = &H50000000
Private Const MIXERCONTROL_CT_CLASS_LIST = &H70000000
'
Private Const MIXERCONTROL_CT_SC_LIST_SINGLE = &H0&
Private Const MIXERCONTROL_CT_SC_SWITCH_BOOLEAN = &H0&
Private Const MIXERCONTROL_CT_SC_LIST_MULTIPLE = &H1000000
'
Private Const MIXERCONTROL_CT_UNITS_BOOLEAN = &H10000
Private Const MIXERCONTROL_CT_UNITS_UNSIGNED = &H30000
'
Private Const MIXERCONTROL_CONTROLTYPE_BOOLEAN = (MIXERCONTROL_CT_CLASS_SWITCH _
                                               Or MIXERCONTROL_CT_SC_SWITCH_BOOLEAN _
                                               Or MIXERCONTROL_CT_UNITS_BOOLEAN)
Private Const MIXERCONTROL_CONTROLTYPE_MUTE = (MIXERCONTROL_CONTROLTYPE_BOOLEAN + 2)
Private Const MIXERCONTROL_CONTROLTYPE_FADER = (MIXERCONTROL_CT_CLASS_FADER _
                                             Or MIXERCONTROL_CT_UNITS_UNSIGNED)
Private Const MIXERCONTROL_CONTROLTYPE_VOLUME = (MIXERCONTROL_CONTROLTYPE_FADER + 1)
Private Const MIXERCONTROL_CONTROLTYPE_SINGLESELECT = (MIXERCONTROL_CT_CLASS_LIST _
                                                    Or MIXERCONTROL_CT_SC_LIST_SINGLE _
                                                    Or MIXERCONTROL_CT_UNITS_BOOLEAN)
Private Const MIXERCONTROL_CONTROLTYPE_MUX = (MIXERCONTROL_CONTROLTYPE_SINGLESELECT + 1)
Private Const MIXERCONTROL_CONTROLTYPE_MULTIPLESELECT = (MIXERCONTROL_CT_CLASS_LIST _
                                                      Or MIXERCONTROL_CT_SC_LIST_MULTIPLE _
                                                      Or MIXERCONTROL_CT_UNITS_BOOLEAN)
Private Const MIXERCONTROL_CONTROLTYPE_MIXER = (MIXERCONTROL_CONTROLTYPE_MULTIPLESELECT + 1)
'
Private Const MIXER_GETCONTROLDETAILSF_VALUE = &H0&
Private Const MIXER_GETCONTROLDETAILSF_LISTTEXT = &H1&
'
Private Const MIXER_SETCONTROLDETAILSF_VALUE = &H0&
'
' Error types:
Private Const MMSYSERR_BASE = 0
Private Const MMSYSERR_NOERROR = 0                          ' no error
Private Const MMSYSERR_ERROR = (MMSYSERR_BASE + 1)          ' unspecified error
Private Const MMSYSERR_BADDEVICEID = (MMSYSERR_BASE + 2)    ' device ID out of range
Private Const MMSYSERR_NOTENABLED = (MMSYSERR_BASE + 3)     ' driver failed enable
Private Const MMSYSERR_ALLOCATED = (MMSYSERR_BASE + 4)      ' device already allocated
Private Const MMSYSERR_INVALHANDLE = (MMSYSERR_BASE + 5)    ' device handle is invalid
Private Const MMSYSERR_NODRIVER = (MMSYSERR_BASE + 6)       ' no device driver present
Private Const MMSYSERR_NOMEM = (MMSYSERR_BASE + 7)          ' memory allocation error
Private Const MMSYSERR_NOTSUPPORTED = (MMSYSERR_BASE + 8)   ' function isn't supported
Private Const MMSYSERR_BADERRNUM = (MMSYSERR_BASE + 9)      ' error value out of range
Private Const MMSYSERR_INVALFLAG = (MMSYSERR_BASE + 10)     ' invalid flag passed
Private Const MMSYSERR_INVALPARAM = (MMSYSERR_BASE + 11)    ' invalid parameter passed
Private Const MMSYSERR_HANDLEBUSY = (MMSYSERR_BASE + 12)    ' handle being used
Private Const MMSYSERR_INVALIDALIAS = (MMSYSERR_BASE + 13)  ' "Specified alias not found in WIN.INI
Private Const MMSYSERR_LASTERROR = (MMSYSERR_BASE + 13)     ' last error in range
'
Private Const WAVERR_BASE = 32
Private Const WAVERR_BADFORMAT = (WAVERR_BASE + 0)          ' unsupported wave format
Private Const WAVERR_STILLPLAYING = (WAVERR_BASE + 1)       ' still something playing
Private Const WAVERR_UNPREPARED = (WAVERR_BASE + 2)         ' header not prepared
Private Const WAVERR_SYNC = (WAVERR_BASE + 3)               ' device is synchronous
'
Private Const MIXERR_BASE = 1024
Private Const MIXERR_INVALLINE = (MIXERR_BASE + 0)          ' the audio line reference is invalid
Private Const MIXERR_INVALCONTROL = (MIXERR_BASE + 1)       ' the control reference is invalid.
Private Const MIXERR_INVALVALUE = (MIXERR_BASE + 2)         ' The value specified for the mixer control is not within the limits for the control.
Private Const MIXERR_LASTERROR = (MIXERR_BASE + 2)
'
'-- Strutture per API: -----------------------------------------------------------------
'
Private Type WAVEOUTCAPS
    wMid As Integer
    wPid As Integer
    vDriverVersion As Long
    szPname(MAXPNAMELEN - 1) As Byte
    dwFormats As Long
    wChannels As Integer
    wReserved1 As Integer
    dwSupport As Long
End Type
'
Private Type WAVEINCAPS
    wMid As Integer
    wPid As Integer
    vDriverVersion As Long
    szPname(MAXPNAMELEN - 1) As Byte
    dwFormats As Long
    wChannels As Integer
    wReserved1 As Integer
End Type
'
Private Type MIXERCAPS
     wMid As Integer                    ' manufacturer id
     wPid As Integer                    ' product id
     vDriverVersion As Long             ' version of the driver
     szPname(MAXPNAMELEN - 1) As Byte   ' product name
     fdwSupport As Long                 ' misc. support bits
     cDestinations As Long              ' count of destinations
End Type
'
Private Type MIXERLINE
    cbStruct As Long
    dwDestination As Long
    dwSource As Long
    dwLineID As Long
    fdwLine As Long
    dwUser As Long
    dwComponentType As Long
    cChannels As Long
    cConnections As Long
    cControls As Long
    szShortName(MIXER_SHORT_NAME_CHARS - 1) As Byte
    szName(MIXER_LONG_NAME_CHARS - 1) As Byte
    dwType As Long
    dwDeviceID As Long
    wMid  As Integer
    wPid As Integer
    vDriverVersion As Long
    szPname(MAXPNAMELEN - 1) As Byte
End Type
'
Private Type MIXERCONTROL
    cbStruct As Long            ' size in Byte of MIXERCONTROL
    dwControlID As Long         ' unique control id for mixer device
    dwControlType As Long       ' MIXERCONTROL_CONTROLTYPE_xxx
    fdwControl As Long          ' MIXERCONTROL_CONTROLF_xxx
    cMultipleItems As Long      ' if MIXERCONTROL_CONTROLF_MULTIPLE set
    szShortName(MIXER_SHORT_NAME_CHARS - 1) As Byte
    szName(MIXER_LONG_NAME_CHARS - 1) As Byte
    Bounds(1 To 6) As Long      ' Longest member of the Bounds union
    Metrics(1 To 6) As Long     ' Longest member of the Metrics union
End Type
'
Private Type MIXERLINECONTROLS
    cbStruct As Long            ' size in Byte of MIXERLINECONTROLS
    dwLineID As Long            ' line id (from MIXERLINE.dwLineID)
    dwControl As Long           ' dwControlID|dwControlType: used with MIXER_GETLINECONTROLSF_ONEBYTYPE or MIXER_GETLINECONTROLSF_ONEBYID
    cControls As Long           ' count of controls pmxctrl points to
    cbmxctrl As Long            ' size in Byte of _one_ MIXERCONTROL
    pamxctrl As Long            ' pointer to first MIXERCONTROL array
End Type
'
Private Type MIXERCONTROLDETAILS
    cbStruct As Long            ' size in Byte of MIXERCONTROLDETAILS
    dwControlID As Long         ' control id to get/set details on
    cChannels As Long           ' number of channels in paDetails array
    item As Long                ' hwndOwner or cMultipleItems
    cbDetails As Long           ' size of _one_ details_XX struct
    paDetails As Long           ' pointer to array of details_XX structs
End Type
'
Private Type MIXERCONTROLDETAILS_BOOLEAN
    fValue As Long
End Type
'
Private Type MIXERCONTROLDETAILS_LISTTEXT
    dwParam1 As Long
    dwParam2 As Long
    szName(MIXER_LONG_NAME_CHARS - 1) As Byte
End Type
'
Private Type MIXERCONTROLDETAILS_UNSIGNED
    dwValue As Long
End Type
'
'-- API: -------------------------------------------------------------------------------
'
Private Declare Function waveInGetNumDevs Lib "winmm.dll" () As Long
Private Declare Function waveInGetDevCaps Lib "winmm.dll" Alias "waveInGetDevCapsA" _
    (ByVal uDeviceID As Long, ByVal WaveOutCapsPointer As Long, _
    ByVal WaveOutCapsStructSize As Long) As Long
'
Private Declare Function waveOutGetNumDevs Lib "winmm.dll" () As Long
Private Declare Function waveOutGetDevCaps Lib "winmm.dll" Alias "waveOutGetDevCapsA" _
    (ByVal uDeviceID As Long, ByVal WaveOutCapsPointer As Long, _
    ByVal WaveOutCapsStructSize As Long) As Long
'
Private Declare Function mixerOpen Lib "winmm.dll" (phmx As Long, ByVal uMxId As Long, _
    ByVal dwCallback As Long, ByVal dwInstance As Long, ByVal fdwOpen As Long) As Long
Private Declare Function mixerClose Lib "winmm.dll" (ByVal hmx As Long) As Long
'
Private Declare Function mixerGetNumDevs Lib "winmm.dll" () As Long
'
Private Declare Function mixerGetDevCaps Lib "winmm.dll" Alias "mixerGetDevCapsA" _
    (ByVal uMxId As Long, pmxcaps As MIXERCAPS, ByVal cbmxcaps As Long) As Long
'
Private Declare Function mixerGetLineInfo Lib "winmm.dll" _
    Alias "mixerGetLineInfoA" (ByVal hmxobj As Long, pmxl As MIXERLINE, _
    ByVal fdwInfo As Long) As Long
'
Private Declare Function mixerGetLineControls Lib "winmm.dll" _
    Alias "mixerGetLineControlsA" (ByVal hmxobj As Long, pmxlc As MIXERLINECONTROLS, _
    ByVal fdwControls As Long) As Long
'
Private Declare Function mixerGetControlDetails Lib "winmm.dll" _
    Alias "mixerGetControlDetailsA" (ByVal hmxobj As Long, _
    pmxcd As MIXERCONTROLDETAILS, ByVal fdwDetails As Long) As Long
'
Private Declare Function mixerSetControlDetails Lib "winmm.dll" (ByVal hmxobj As Long, _
    pmxcd As MIXERCONTROLDETAILS, ByVal fdwDetails As Long) As Long
'
'-- VerificaSchedaAudioInCap, VerificaSchedaAudioOutCap: -------------------------------
'
Private Const WAVE_FORMAT_PCM% = 1
Private Const WAVE_FORMAT_QUERY& = &H1
'
Private Type WAVEFORMATEX
    wFormatTag As Integer
    nChannels As Integer
    nSamplesPerSec As Long
    nAvgBytesPerSec As Long
    nBlockAlign As Integer
    wBitsPerSample As Integer
    cbSize As Integer
End Type
'
Private Declare Function waveInOpen Lib "winmm.dll" _
    (lphWaveOut As Long, ByVal uDeviceID As Long, _
    ByVal WaveFormatExPointer As Long, ByVal dwCallback As Long, _
    ByVal dwInstance As Long, ByVal dwFlags As Long) As Long
'
Private Declare Function waveOutOpen Lib "winmm.dll" _
    (lphWaveOut As Long, ByVal uDeviceID As Long, _
    ByVal WaveFormatExPointer As Long, ByVal dwCallback As Long, _
    ByVal dwInstance As Long, ByVal dwFlags As Long) As Long

Public Function CercaSchedeAudio() As Boolean
'
'   Cerca tutte le schede audio (mixers) presenti su questo PC.
'   Nota: alcune schede audio (e.g. la Realtek HD Audio) appaiono come
'   sdoppiate in "Playback mixer device" e "Recording mixer device".
'
    Dim mxGND&, uMixerID&, hmixer&, cDestinations&, lRet&
    Dim pmxcaps As MIXERCAPS
'
    On Error GoTo CercaSchedeAudio_ERR
'
    NSchedeAudio = 0
    mxGND = mixerGetNumDevs
    For uMixerID = 0 To mxGND - 1
        lRet = mixerOpen(hmixer, uMixerID, 0, 0, 0)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerOpen #" & uMixerID & " - Err: " & WaveERR(lRet)
        End If
'
        lRet = mixerGetDevCaps(uMixerID, pmxcaps, Len(pmxcaps))
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerGetDevCaps #" & uMixerID & " - Err: " & WaveERR(lRet)
        End If
'
        cDestinations = pmxcaps.cDestinations   ' Solo per verifiche.
'
        NSchedeAudio = NSchedeAudio + 1
        ReDim Preserve SchedeAudio(1 To NSchedeAudio)
'
        SchedeAudio(NSchedeAudio) = BytesToString(pmxcaps.szPname)
'
        lRet = mixerClose(hmixer)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerClose #" & uMixerID & " - Err: " & WaveERR(lRet)
        End If
    Next uMixerID
'
'
CercaSchedeAudio_ERR:
    CercaSchedeAudio = (Err.Number = 0) And (0 < NSchedeAudio)
'
    If (Err.Number <> 0) Then
        mixerClose hmixer
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: CercaSchedeAudio"
        Err.Clear
    End If
'
'
'
End Function
Public Function CercaMixersWi() As Boolean
'
'   Cerca i Mixers, con una destinazione di tipo WAVEIN,
'   presenti sulle schede audio di questo PC:
'
    Dim MxI&, mxGND&, uMxId&, Pname$, lRet&
    Dim mxcaps As MIXERCAPS, mxl As MIXERLINE
    Dim uD&, wIGND&, pwoc As WAVEINCAPS
'
    On Error GoTo CercaMixersWi_ERR
'
    NMixWi = 0
'
    mxGND = mixerGetNumDevs
    For uMxId = 0 To mxGND - 1
        lRet = mixerOpen(hmixer, uMxId, 0, 0, 0)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerOpen #" & uMxId & " - Err: " & WaveERR(lRet)
        End If
'
        mxl.cbStruct = Len(mxl)
        mxl.dwComponentType = MIXERLINE_COMPONENTTYPE_DST_WAVEIN
        lRet = mixerGetLineInfo(hmixer, mxl, MIXER_GETLINEINFOF_COMPONENTTYPE)
'
        If (lRet = MMSYSERR_NOERROR) Then
            lRet = mixerGetDevCaps(uMxId, mxcaps, Len(mxcaps))
            If (lRet <> MMSYSERR_NOERROR) Then
                Err.Raise 1001, , "mixerGetDevCaps #" & uMxId & " - Err: " & WaveERR(lRet)
            End If
'
            NMixWi = NMixWi + 1
            ReDim Preserve MixerWi(1 To NMixWi)
'
            MixerWi(NMixWi).MixerID = uMxId
            MixerWi(NMixWi).Name = BytesToString(mxcaps.szPname)
'
            ' Cerco le linee del mixer:
            Call GetMixerLines(MixerWi(NMixWi), MIXERLINE_COMPONENTTYPE_DST_WAVEIN)
        End If
'
        lRet = mixerClose(hmixer)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerClose #" & uMxId & " - Err: " & WaveERR(lRet)
        End If
    Next uMxId
'
    If (NMixWi = 0) Then
        Err.Raise 1001, , "CercaMixersWi - Err: (NMixWi = 0)"
    End If
'
    ' Assegno gli identificativi (da usare
    ' in WaveInOpen) delle schede audio
    ' contenenti i mixers trovati:
    wIGND = waveInGetNumDevs
    For uD = 0 To wIGND - 1
        lRet = waveInGetDevCaps(uD, VarPtr(pwoc), Len(pwoc))
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "waveInGetDevCaps - Err: " & WaveERR(lRet)
        End If
'
        Pname$ = BytesToString(pwoc.szPname)
'
        For MxI = 1 To NMixWi
            If (MixerWi(MxI).Name = Pname$) Then
                MixerWi(MxI).SchedaAudioID = uD
            End If
        Next MxI
    Next uD
'
'
CercaMixersWi_ERR:
    CercaMixersWi = (Err.Number = 0)
'
    If (Err.Number <> 0) Then
        mixerClose hmixer
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: CercaMixersWi"
        Err.Clear
    End If
'
'
'
End Function
Public Function CercaMixersSp() As Boolean
'
'   Cerca i Mixers, con una destinazione di tipo SPEAKERS,
'   presenti sulle schede audio di questo PC:
'
    Dim MxI&, mxGND&, uMxId&, Pname$, lRet&
    Dim mxcaps As MIXERCAPS, mxl As MIXERLINE
    Dim uD&, wOGND&, pwoc As WAVEOUTCAPS
'
    On Error GoTo CercaMixersSp_ERR
'
    NMixSp = 0
'
    mxGND = mixerGetNumDevs
    For uMxId = 0 To mxGND - 1
        lRet = mixerOpen(hmixer, uMxId, 0, 0, 0)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerOpen #" & uMxId & " - Err: " & WaveERR(lRet)
        End If
'
        mxl.cbStruct = Len(mxl)
        mxl.dwComponentType = MIXERLINE_COMPONENTTYPE_DST_SPEAKERS
        lRet = mixerGetLineInfo(hmixer, mxl, MIXER_GETLINEINFOF_COMPONENTTYPE)
'
        If (lRet = MMSYSERR_NOERROR) Then
            lRet = mixerGetDevCaps(uMxId, mxcaps, Len(mxcaps))
            If (lRet <> MMSYSERR_NOERROR) Then
                Err.Raise 1001, , "mixerGetDevCaps #" & uMxId & " - Err: " & WaveERR(lRet)
            End If
'
            NMixSp = NMixSp + 1
            ReDim Preserve MixerSp(1 To NMixSp)
'
            MixerSp(NMixSp).MixerID = uMxId
            MixerSp(NMixSp).Name = BytesToString(mxcaps.szPname)
'
            ' Cerco le linee del mixer:
            Call GetMixerLines(MixerSp(NMixSp), MIXERLINE_COMPONENTTYPE_DST_SPEAKERS)
        End If
'
        lRet = mixerClose(hmixer)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerClose #" & uMxId & " - Err: " & WaveERR(lRet)
        End If
    Next uMxId
'
    If (NMixSp = 0) Then
        Err.Raise 1001, , "CercaMixersSp - Err: (NMixSp = 0)"
    End If
'
    ' Assegno gli identificativi (da usare
    ' in WaveOutOpen) delle schede audio
    ' contenenti i mixers trovati:
    wOGND = waveOutGetNumDevs
    For uD = 0 To wOGND - 1
        lRet = waveOutGetDevCaps(uD, VarPtr(pwoc), Len(pwoc))
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "waveOutGetDevCaps #" & uD & " - Err: " & WaveERR(lRet)
        End If
'
        Pname$ = BytesToString(pwoc.szPname)
'
        For MxI = 1 To NMixSp
            If (MixerSp(MxI).Name = Pname$) Then
                MixerSp(MxI).SchedaAudioID = uD
            End If
        Next MxI
    Next uD
'
'
CercaMixersSp_ERR:
    CercaMixersSp = (Err.Number = 0)
'
    If (Err.Number <> 0) Then
        mixerClose hmixer
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: CercaMixersSp"
        Err.Clear
    End If
'
'
'
End Function
Public Sub ImpostaAbilitazioneLinea(ByRef Mixer_I As Mixer_Type, _
    Optional ByVal Ich As Long, Optional ByVal bIsDestLine As Boolean = False)
'
'   Attiva (/disattiva), per "Playback" o "Recording", una linea del mixer Mixer_I.
'   Se bIsDestLine = True viene impostata l' abilitazione della "Destination line";
'   se no quella della "Source line" Ich.
'   Se la "Destination line" ha un controllo di tipo MUX la linea viene selezionata;
'   se no (linee con controllo "Mute") lo stato della linea viene commutato:
'
'
    Dim lRet&
'
    On Error GoTo ImpostaAbilitazioneLinea_ERR
'
    lRet = mixerOpen(hmixer, Mixer_I.MixerID, 0, 0, 0)
    If (lRet <> MMSYSERR_NOERROR) Then
        Err.Raise 1001, , "mixerOpen - Err: " & WaveERR(lRet)
    End If
'
    If bIsDestLine Then
        SetDstLineStateMute Mixer_I
'
    ElseIf Mixer_I.bDstIsMUX Then
        SetSrcLineStateMUX Mixer_I, Ich
'
    Else
        SetSrcLineStateMute Mixer_I, Ich
    End If
'
'
ImpostaAbilitazioneLinea_ERR:
    lRet = mixerClose(hmixer)
'
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: ImpostaAbilitazioneLinea"
        Err.Clear
    End If
'
'
'
End Sub
Private Sub GetMixerLines(ByRef Mixer_I As Mixer_Type, ByVal ComponentType As Long)
'
'   Trova per il mixer Mixer_I la "Destination line", le "Source lines"
'   ed i controlli a loro associati:
'
    Dim Ich&, IDst&, lRet&
'
    Dim mxl As MIXERLINE
    Dim mxlc As MIXERLINECONTROLS
    Dim mxc As MIXERCONTROL
'
    On Error GoTo GetMixerLines_ERR
'
    ' Cerco i parametri della "Destination line" di tipo ComponentType:
    mxl.cbStruct = Len(mxl)
    mxl.dwComponentType = ComponentType ' MIXERLINE_COMPONENTTYPE_DST_SPEAKERS | MIXERLINE_COMPONENTTYPE_DST_WAVEIN.
'
    lRet = mixerGetLineInfo(hmixer, mxl, MIXER_GETLINEINFOF_COMPONENTTYPE)
    If (lRet <> MMSYSERR_NOERROR) Then
        Err.Raise 1001, , "mixerGetLineInfo 1 - Err: " & WaveERR(lRet)
    End If
'
    Mixer_I.DstLine.LineID = mxl.dwLineID
    Mixer_I.DstLine.Name = BytesToString(mxl.szName)
    Mixer_I.DstLine.Controls = mxl.cControls
    Mixer_I.DstLine.Channels = mxl.cChannels
'
    Call GetDstLineStateMute(Mixer_I)   ' Stato "Mute".
    Call GetLineVolume(mxl.dwLineID, Mixer_I.DstLine.Volume, _
                                     Mixer_I.DstLine.VolumeStep, _
                                     Mixer_I.DstLine.VolumeName)
'
    IDst = mxl.dwDestination
    Mixer_I.NSrcLines = mxl.cConnections
    ReDim Mixer_I.SrcLine(0 To Mixer_I.NSrcLines - 1)
'
    ' Interrogo la "Destination line" sulla presenza,
    ' o meno, di un controllo di tipo MUX:
    mxlc.cbStruct = Len(mxlc)
    mxlc.dwLineID = mxl.dwLineID
    mxlc.cControls = 1
    mxlc.dwControl = MIXERCONTROL_CONTROLTYPE_MUX
    mxlc.pamxctrl = VarPtr(mxc)
    mxlc.cbmxctrl = Len(mxc)
'
    lRet = mixerGetLineControls(hmixer, mxlc, MIXER_GETLINECONTROLSF_ONEBYTYPE)
    If (lRet = MMSYSERR_NOERROR) Then
        ' La "Destination line", ha un controllo MUX:
        Mixer_I.bDstIsMUX = True
        Mixer_I.MUXControlID = mxc.dwControlID
        Mixer_I.MUXControlItems = mxc.cMultipleItems
    Else
        Mixer_I.bDstIsMUX = False
    End If
'
'---------------------------------------------------------------------------------------
'
    ' Cerco gli ID i nomi ed i volumi delle "Source lines":
    For Ich = 0 To Mixer_I.NSrcLines - 1
        mxl.cbStruct = Len(mxl)
        mxl.dwDestination = IDst
        mxl.dwSource = Ich
        lRet = mixerGetLineInfo(hmixer, mxl, MIXER_GETLINEINFOF_SOURCE)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerGetLineInfo 2 - Err: " & WaveERR(lRet)
        End If
'
        Mixer_I.SrcLine(Ich).LineID = mxl.dwLineID
        Mixer_I.SrcLine(Ich).Name = BytesToString(mxl.szName)
        Mixer_I.SrcLine(Ich).Controls = mxl.cControls
        Mixer_I.SrcLine(Ich).Channels = mxl.cChannels
'
        Call GetLineVolume(mxl.dwLineID, Mixer_I.SrcLine(Ich).Volume, _
                                         Mixer_I.SrcLine(Ich).VolumeStep, _
                                         Mixer_I.SrcLine(Ich).VolumeName)
    Next Ich
'
    ' Cerco lo stato (Mute | Select) delle "Source lines":
    If Mixer_I.bDstIsMUX Then
        Call GetSrcLineStateMUX(Mixer_I)    ' Stato "Select".
    Else
        Call GetSrcLineStateMute(Mixer_I)   ' Stato "Mute".
    End If
'
'
GetMixerLines_ERR:
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: GetMixerLines"
        Err.Clear
    End If
'
'
'
End Sub
Private Function GetLineVolume(ByVal LineID As Long, ByRef Volume As Long, _
   ByRef VolumeStep As Long, ByRef VolumeName$) As Boolean
'
'   Trova il livello di volume impostato per la linea con identificativo LineID;
'   ritorna anche in VolumeStep la minima variazione possibile di valore ed in
'   VolumeName$ il nome di questo controllo.  Se il controllo di volume non e'
'   disponibile viene ritornato Volume = -1:
'
    Dim lRet&, MaxVol&
'
    Dim mxc As MIXERCONTROL
    Dim mxlc As MIXERLINECONTROLS
    Dim mxcd As MIXERCONTROLDETAILS
    Dim mxcd_u As MIXERCONTROLDETAILS_UNSIGNED
'
    On Error GoTo GetLineVolume_ERR
'
    mxlc.cbStruct = Len(mxlc)
    mxlc.dwLineID = LineID
    mxlc.cControls = 1
    mxlc.dwControl = MIXERCONTROL_CONTROLTYPE_VOLUME
    mxlc.pamxctrl = VarPtr(mxc)
    mxlc.cbmxctrl = Len(mxc)
'
    lRet = mixerGetLineControls(hmixer, mxlc, MIXER_GETLINECONTROLSF_ONEBYTYPE)
'
    If (lRet = MMSYSERR_NOERROR) Then
        mxcd.cbStruct = Len(mxcd)
        mxcd.dwControlID = mxc.dwControlID
        mxcd.cChannels = 1
        mxcd.item = 0
        mxcd.paDetails = VarPtr(mxcd_u)
        mxcd.cbDetails = Len(mxcd_u)
'
        lRet = mixerGetControlDetails(hmixer, mxcd, MIXER_GETCONTROLDETAILSF_VALUE)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerGetControlDetails - Err: " & WaveERR(lRet)
        End If
'
        VolumeName$ = BytesToString(mxc.szName)
        MaxVol = mxc.Bounds(2)                  ' Per questo tipo di controllo
                                                ' e' sempre MaxVol = 65535.
'
        Volume = mxcd_u.dwValue
        VolumeStep = MaxVol / mxc.Metrics(1)
'
    Else
        Volume = -1  ' "Volume control" non disponibile.
    End If
'
'
GetLineVolume_ERR:
    GetLineVolume = (Err.Number = 0)
'
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: GetLineVolume"
        Err.Clear
    End If
'
'
'
End Function
Private Function GetSrcLineStateMute(ByRef Mixer_I As Mixer_Type) As Boolean
'
'   Cerca lo stato di abilitazione (Mute) delle "Source lines":
'
    Dim Ich&, lRet&
'
    Dim mxc As MIXERCONTROL
    Dim mxlc As MIXERLINECONTROLS
    Dim mxcd As MIXERCONTROLDETAILS
    Dim mxcd_b As MIXERCONTROLDETAILS_BOOLEAN
'
    On Error GoTo GetSrcLineStateMute_ERR
'
    For Ich = 0 To Mixer_I.NSrcLines - 1
        mxlc.cbStruct = Len(mxlc)
        mxlc.dwLineID = Mixer_I.SrcLine(Ich).LineID
        mxlc.cControls = 1
        mxlc.dwControl = MIXERCONTROL_CONTROLTYPE_MUTE
        mxlc.pamxctrl = VarPtr(mxc)
        mxlc.cbmxctrl = Len(mxc)
'
        lRet = mixerGetLineControls(hmixer, mxlc, MIXER_GETLINECONTROLSF_ONEBYTYPE)
'
        If (lRet = MMSYSERR_NOERROR) Then
            mxcd.cbStruct = Len(mxcd)
            mxcd.dwControlID = mxc.dwControlID
            mxcd.cChannels = 1
            mxcd.item = 0
            mxcd.paDetails = VarPtr(mxcd_b)
            mxcd.cbDetails = Len(mxcd_b)
'
            lRet = mixerGetControlDetails(hmixer, mxcd, MIXER_GETCONTROLDETAILSF_VALUE)
            If (lRet <> MMSYSERR_NOERROR) Then
                Err.Raise 1001, , "mixerGetControlDetails - Err: " & WaveERR(lRet)
            End If
'
            Mixer_I.SrcLine(Ich).MuteControlID = mxc.dwControlID
            Mixer_I.SrcLine(Ich).bNotMute = (Not CBool(mxcd_b.fValue))
            Mixer_I.SrcLine(Ich).bMuteOK = True
'
        Else
            Mixer_I.SrcLine(Ich).bMuteOK = False
        End If
    Next Ich
'
'
GetSrcLineStateMute_ERR:
    GetSrcLineStateMute = (lRet = 0)
'
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: GetSrcLineStateMute"
        Err.Clear
    End If
'
'
'
End Function
Private Function GetDstLineStateMute(ByRef Mixer_I As Mixer_Type) As Boolean
'
'   Cerca lo stato di abilitazione (Mute) di una "Destination line":
'
    Dim lRet&
'
    Dim mxc As MIXERCONTROL
    Dim mxlc As MIXERLINECONTROLS
    Dim mxcd As MIXERCONTROLDETAILS
    Dim mxcd_b As MIXERCONTROLDETAILS_BOOLEAN
'
    On Error GoTo GetDstLineStateMute_ERR
'
    mxlc.cbStruct = Len(mxlc)
    mxlc.dwLineID = Mixer_I.DstLine.LineID
    mxlc.cControls = 1
    mxlc.dwControl = MIXERCONTROL_CONTROLTYPE_MUTE
    mxlc.pamxctrl = VarPtr(mxc)
    mxlc.cbmxctrl = Len(mxc)
'
    lRet = mixerGetLineControls(hmixer, mxlc, MIXER_GETLINECONTROLSF_ONEBYTYPE)
'
    If (lRet = MMSYSERR_NOERROR) Then
        mxcd.cbStruct = Len(mxcd)
        mxcd.dwControlID = mxc.dwControlID
        mxcd.cChannels = 1
        mxcd.item = 0
        mxcd.paDetails = VarPtr(mxcd_b)
        mxcd.cbDetails = Len(mxcd_b)
'
        lRet = mixerGetControlDetails(hmixer, mxcd, MIXER_GETCONTROLDETAILSF_VALUE)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerGetControlDetails - Err: " & WaveERR(lRet)
        End If
'
        Mixer_I.DstLine.MuteControlID = mxc.dwControlID
        Mixer_I.DstLine.bNotMute = (Not CBool(mxcd_b.fValue))
        Mixer_I.DstLine.bMuteOK = True
'
    Else
        Mixer_I.DstLine.bMuteOK = False
    End If
'
'
GetDstLineStateMute_ERR:
    GetDstLineStateMute = (lRet = 0)
'
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: GetDstLineStateMute"
        Err.Clear
    End If
'
'
'
End Function
Private Sub SetSrcLineStateMute(ByRef Mixer_I As Mixer_Type, ByVal Ich As Long)
'
'   Commuta lo stato di abilitazione (Mute) di una "Source line":
'
    Dim lRet&
'
    Dim mxc As MIXERCONTROL
    Dim mxlc As MIXERLINECONTROLS
    Dim mxcd As MIXERCONTROLDETAILS
    Dim mxcd_b As MIXERCONTROLDETAILS_BOOLEAN
'
    On Error GoTo SetSrcLineStateMute_ERR
'
    mxlc.cbStruct = Len(mxlc)
    mxlc.dwLineID = Mixer_I.SrcLine(Ich).LineID
    mxlc.cControls = 1
    mxlc.dwControl = MIXERCONTROL_CONTROLTYPE_MUTE
    mxlc.pamxctrl = VarPtr(mxc)
    mxlc.cbmxctrl = Len(mxc)
'
    lRet = mixerGetLineControls(hmixer, mxlc, MIXER_GETLINECONTROLSF_ONEBYTYPE)
'
    If (lRet = MMSYSERR_NOERROR) Then
        ' Commuto lo stato della linea Ich:
        Mixer_I.SrcLine(Ich).bNotMute = (Not Mixer_I.SrcLine(Ich).bNotMute)
'
        mxcd.cbStruct = Len(mxcd)
        mxcd.dwControlID = mxc.dwControlID
        mxcd.cChannels = 1
        mxcd.item = 0
        mxcd.paDetails = VarPtr(mxcd_b)
        mxcd.cbDetails = Len(mxcd_b)
        mxcd_b.fValue = IIf(Mixer_I.SrcLine(Ich).bNotMute, 0, 1)
'
        lRet = mixerSetControlDetails(hmixer, mxcd, MIXER_SETCONTROLDETAILSF_VALUE)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerSetControlDetails - Err: " & WaveERR(lRet)
        End If
    End If
'
'
SetSrcLineStateMute_ERR:
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: SetSrcLineStateMute"
        Err.Clear
    End If
'
'
'
End Sub
Private Sub SetDstLineStateMute(ByRef Mixer_I As Mixer_Type)
'
'   Commuta lo stato di abilitazione (Mute) di una "Source line":
'
    Dim lRet&
'
    Dim mxc As MIXERCONTROL
    Dim mxlc As MIXERLINECONTROLS
    Dim mxcd As MIXERCONTROLDETAILS
    Dim mxcd_b As MIXERCONTROLDETAILS_BOOLEAN
'
    On Error GoTo SetDstLineStateMute_ERR
'
    mxlc.cbStruct = Len(mxlc)
    mxlc.dwLineID = Mixer_I.DstLine.LineID
    mxlc.cControls = 1
    mxlc.dwControl = MIXERCONTROL_CONTROLTYPE_MUTE
    mxlc.pamxctrl = VarPtr(mxc)
    mxlc.cbmxctrl = Len(mxc)
'
    lRet = mixerGetLineControls(hmixer, mxlc, MIXER_GETLINECONTROLSF_ONEBYTYPE)
'
    If (lRet = MMSYSERR_NOERROR) Then
        ' Commuto lo stato della linea Ich:
        Mixer_I.DstLine.bNotMute = (Not Mixer_I.DstLine.bNotMute)
'
        mxcd.cbStruct = Len(mxcd)
        mxcd.dwControlID = mxc.dwControlID
        mxcd.cChannels = 1
        mxcd.item = 0
        mxcd.paDetails = VarPtr(mxcd_b)
        mxcd.cbDetails = Len(mxcd_b)
        mxcd_b.fValue = IIf(Mixer_I.DstLine.bNotMute, 0, 1)
'
        lRet = mixerSetControlDetails(hmixer, mxcd, MIXER_SETCONTROLDETAILSF_VALUE)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerSetControlDetails - Err: " & WaveERR(lRet)
        End If
    End If
'
'
SetDstLineStateMute_ERR:
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: SetDstLineStateMute"
        Err.Clear
    End If
'
'
'
End Sub
Private Function GetSrcLineStateMUX(ByRef Mixer_I As Mixer_Type) As Boolean
'
'   Cerca lo stato di abilitazione (Select) delle "Source lines"
'   collegate ad un controllo di tipo MUX:
'
    Dim I&, J&, lRet&, Name$
'
    Dim mxlc As MIXERLINECONTROLS
    Dim mxcd As MIXERCONTROLDETAILS
    Dim mxcd_l() As MIXERCONTROLDETAILS_LISTTEXT
    Dim mxcd_b() As MIXERCONTROLDETAILS_BOOLEAN
'
    On Error GoTo GetSrcLineStateMUX_ERR
'
    ' Cerco il nome delle "Source lines" collegate al MUX:
    ReDim mxcd_l(0 To Mixer_I.MUXControlItems - 1) As MIXERCONTROLDETAILS_LISTTEXT
'
    mxcd.cbStruct = Len(mxcd)
    mxcd.dwControlID = Mixer_I.MUXControlID
    mxcd.cChannels = 1
    mxcd.item = Mixer_I.MUXControlItems
    mxcd.cbDetails = Len(mxcd_l(0))
    mxcd.paDetails = VarPtr(mxcd_l(0))
'
    lRet = mixerGetControlDetails(hmixer, mxcd, MIXER_GETCONTROLDETAILSF_LISTTEXT)
    If (lRet <> MMSYSERR_NOERROR) Then
        Err.Raise 1001, , "mixerGetControlDetails 1 - Err: " & WaveERR(lRet)
    End If
'
    ' Cerco lo stato delle "Source lines" collegate al MUX:
    ReDim mxcd_b(0 To Mixer_I.MUXControlItems - 1) As MIXERCONTROLDETAILS_BOOLEAN
'
    mxcd.cbStruct = Len(mxcd)
    mxcd.dwControlID = Mixer_I.MUXControlID
    mxcd.cChannels = 1
    mxcd.item = Mixer_I.MUXControlItems
    mxcd.cbDetails = Len(mxcd_b(0))
    mxcd.paDetails = VarPtr(mxcd_b(0))
'
    lRet = mixerGetControlDetails(hmixer, mxcd, MIXER_GETCONTROLDETAILSF_VALUE)
    If (lRet <> MMSYSERR_NOERROR) Then
        Err.Raise 1001, , "mixerGetControlDetails 2 - Err: " & WaveERR(lRet)
    End If
'
    ' Cerco la corrispondenza fra gli indici delle "Source lines",
    ' come elencate in Mixer_I.SrcLine(), e gli indici delle stesse,
    ' come elencate in mxcd_l(), per poter assegnare i valori di stato
    ' in ordine corretto; necessario, perche' i controlli MUX di alcune
    ' schede audio usano uma corrispondenza inversa e/o non sequenziale:
    For I = 0 To Mixer_I.MUXControlItems - 1
        Name$ = BytesToString(mxcd_l(I).szName)
'
        For J = 0 To Mixer_I.NSrcLines - 1
            If (Mixer_I.SrcLine(J).Name = Name$) Then
                Mixer_I.SrcLine(J).bNotMute = CBool(mxcd_b(I).fValue)
                Mixer_I.SrcLine(J).bMuteOK = True
                Mixer_I.SrcLine(J).IrMUX = I
            End If
        Next J
    Next I
'
'
GetSrcLineStateMUX_ERR:
    GetSrcLineStateMUX = (Err.Number = 0)
'
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: GetSrcLineStateMUX"
        Err.Clear
    End If
'
'
'
End Function
Private Sub SetSrcLineStateMUX(ByRef Mixer_I As Mixer_Type, ByVal Ich As Long)
'
'   Attiva lo stato di abilitazione (Select) della "Source line" Ich
'   collegata ad un controllo di tipo MUX:
'
    Dim I&, lRet&
'
    Dim mxlc As MIXERLINECONTROLS
    Dim mxcd As MIXERCONTROLDETAILS
    Dim mxcd_b() As MIXERCONTROLDETAILS_BOOLEAN
'
    On Error GoTo SetSrcLineStateMUX_ERR
'
    ReDim mxcd_b(0 To Mixer_I.MUXControlItems - 1) As MIXERCONTROLDETAILS_BOOLEAN
'
    mxcd.cbStruct = Len(mxcd)
    mxcd.dwControlID = Mixer_I.MUXControlID
    mxcd.cChannels = 1
    mxcd.item = Mixer_I.MUXControlItems
    mxcd.cbDetails = Len(mxcd_b(0))
    mxcd.paDetails = VarPtr(mxcd_b(0))
'
    ' Attivo la linea Ich e disattivo tutte le altre:
    For I = 0 To Mixer_I.NSrcLines - 1
        If (I = Ich) Then
            Mixer_I.SrcLine(I).bNotMute = True
            mxcd_b(Mixer_I.SrcLine(I).IrMUX).fValue = 1
        Else
            Mixer_I.SrcLine(I).bNotMute = False
            mxcd_b(Mixer_I.SrcLine(I).IrMUX).fValue = 0
        End If
    Next I
'
    lRet = mixerSetControlDetails(hmixer, mxcd, MIXER_SETCONTROLDETAILSF_VALUE)
    If (lRet <> MMSYSERR_NOERROR) Then
        Err.Raise 1001, , "mixerSetControlDetails - Err: " & WaveERR(lRet)
    End If
'
'
SetSrcLineStateMUX_ERR:
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: SetSrcLineStateMUX"
        Err.Clear
    End If
'
'
'
End Sub
Public Function VerificaSchedaAudioOutCap(ByVal uDeviceID_V As Long, _
    ByVal Fs_V As Long, Optional ByVal NCanali_V As Integer = 2, _
    Optional ByVal BitsPerSample_V As Integer = 16) As Boolean
'
'   Verifica che la frequenza di campionamento Fs_V, il numero di canali
'   NCanali_V e la risoluzione BitsPerSample_V siano supportati dalla
'   scheda audio uDeviceID_V in modalita' "Playback":
'
    Dim phwo&, lRet&, FormatWaveOut As WAVEFORMATEX
'
    ' Imposto lo stato da verificare:
    With FormatWaveOut
        .wFormatTag = WAVE_FORMAT_PCM
        .nChannels = NCanali_V
        .nSamplesPerSec = Fs_V ' [Hz]
        .wBitsPerSample = BitsPerSample_V
        .nBlockAlign = (.nChannels * .wBitsPerSample) \ 8
        .nAvgBytesPerSec = .nBlockAlign * .nSamplesPerSec
        .cbSize = 0
    End With
'
    ' Interrogo il collegamento con la scheda audio:
    lRet = waveOutOpen(phwo, uDeviceID_V, VarPtr(FormatWaveOut), 0, 0, WAVE_FORMAT_QUERY)
    VerificaSchedaAudioOutCap = (lRet = MMSYSERR_NOERROR)
'
'
'
End Function
Public Function VerificaSchedaAudioInCap(ByVal uDeviceID_V As Long, _
    ByVal Fs_V As Long, Optional ByVal NCanali_V As Integer = 2, _
    Optional ByVal BitsPerSample_V As Integer = 16) As Boolean
'
'   Verifica che la frequenza di campionamento Fs_V, il numero di canali
'   NCanali_V e la risoluzione BitsPerSample_V siano supportati dalla
'   scheda audio uDeviceID_V in modalita' "Recording":
'
    Dim phwi&, lRet&, FormatWaveIn As WAVEFORMATEX
'
    ' Imposto lo stato da verificare:
    With FormatWaveIn
        .wFormatTag = WAVE_FORMAT_PCM
        .nChannels = NCanali_V
        .nSamplesPerSec = Fs_V ' [Hz]
        .wBitsPerSample = BitsPerSample_V
        .nBlockAlign = (.nChannels * .wBitsPerSample) \ 8
        .nAvgBytesPerSec = .nBlockAlign * .nSamplesPerSec
        .cbSize = 0
    End With
'
    ' Interrogo il collegamento con la scheda audio:
    lRet = waveInOpen(phwi, uDeviceID_V, VarPtr(FormatWaveIn), 0, 0, WAVE_FORMAT_QUERY)
    VerificaSchedaAudioInCap = (lRet = MMSYSERR_NOERROR)
'
'
'
End Function
Public Function CercaWaveInFs(Mixer_I As Mixer_Type, ByRef FsList() As Long, _
    Optional ByVal NCanali_V As Integer = 2, _
    Optional ByVal BitsPerSample_V As Integer = 16) As Long
'
'   Ritorna, per i parametri specificati, il numero NFs di frequenze di
'   campionamento ammesse per il mixer Mixer_I in modalita' "Recording";
'   la lista delle frequenze di campionamento ammesse viene composta nel
'   vettore FsList(1 To NFs).
'   Deve, precedentemente, essere stata chiamata la Function CercaMixersWi:
'
    Dim NFs&, uDeviceID&
'
    On Error GoTo CercaWaveInFs_ERR
'
    NFs = 0
    uDeviceID = Mixer_I.SchedaAudioID
'
    If VerificaSchedaAudioInCap(uDeviceID, 8000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 8000
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 11025, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 11025
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 16000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 16000
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 22050, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 22050
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 24000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 24000
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 48000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 48000
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 88200, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 88200
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 96000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 96000
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 172400, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 172400
    End If
'
    If VerificaSchedaAudioInCap(uDeviceID, 192000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 192000
    End If
'
    CercaWaveInFs = NFs
'
'
CercaWaveInFs_ERR:
    If (Err.Number <> 0) Then
        CercaWaveInFs = 0
'
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: CercaWaveInFs"
        Err.Clear
    End If
'
'
'
End Function
Public Function CercaWaveOutFs(Mixer_I As Mixer_Type, ByRef FsList() As Long, _
    Optional ByVal NCanali_V As Integer = 2, _
    Optional ByVal BitsPerSample_V As Integer = 16) As Long
'
'   Ritorna, per i parametri specificati, il numero NFs di frequenze di
'   campionamento ammesse per il mixer Mixer_I in modalita' "Playback";
'   la lista delle frequenze di campionamento ammesse viene composta nel
'   vettore FsList(1 To NFs).
'   Deve, precedentemente, essere stata chiamata la Function CercaMixersSp:
'
    Dim NFs&, uDeviceID&
'
    On Error GoTo CercaWaveOutFs_ERR
'
    NFs = 0
    uDeviceID = Mixer_I.SchedaAudioID
'
    If VerificaSchedaAudioOutCap(uDeviceID, 8000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 8000
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 11025, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 11025
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 16000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 16000
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 22050, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 22050
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 24000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 24000
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 48000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 48000
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 88200, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 88200
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 96000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 96000
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 172400, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 172400
    End If
'
    If VerificaSchedaAudioOutCap(uDeviceID, 192000, NCanali_V, BitsPerSample_V) Then
        NFs = NFs + 1
        ReDim Preserve FsList(1 To NFs)
        FsList(NFs) = 192000
    End If
'
    CercaWaveOutFs = NFs
'
'
CercaWaveOutFs_ERR:
    If (Err.Number <> 0) Then
        CercaWaveOutFs = 0
'
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: CercaWaveOutFs"
        Err.Clear
    End If
'
'
'
End Function
Public Sub ImpostaVolumeLinea(ByRef Mixer_I As Mixer_Type, ByVal Volume As Long, _
    Optional ByVal Ich As Long, Optional ByVal bIsDestLine As Boolean = False)
'
'   Imposta il Volume di una linea del mixer Mixer_I.
'   Se bIsDestLine = True viene impostato il volume della "Destination line";
'   se no quello della "Source line" Ich:
'
    Dim lRet&
'
    Dim mxc As MIXERCONTROL
    Dim mxlc As MIXERLINECONTROLS
    Dim mxcd As MIXERCONTROLDETAILS
    Dim mxcd_u As MIXERCONTROLDETAILS_UNSIGNED
'
    On Error GoTo ImpostaVolumeLinea_ERR
'
    lRet = mixerOpen(hmixer, Mixer_I.MixerID, 0, 0, 0)
    If (lRet <> MMSYSERR_NOERROR) Then
        Err.Raise 1001, , "mixerOpen - Err: " & WaveERR(lRet)
    End If
'
    mxlc.cbStruct = Len(mxlc)
    If bIsDestLine Then
        mxlc.dwLineID = Mixer_I.DstLine.LineID
    Else
        mxlc.dwLineID = Mixer_I.SrcLine(Ich).LineID
    End If
    mxlc.cControls = 1
    mxlc.dwControl = MIXERCONTROL_CONTROLTYPE_VOLUME
    mxlc.pamxctrl = VarPtr(mxc)
    mxlc.cbmxctrl = Len(mxc)
'
    lRet = mixerGetLineControls(hmixer, mxlc, MIXER_GETLINECONTROLSF_ONEBYTYPE)
'
    If (lRet = MMSYSERR_NOERROR) Then
        mxcd.cbStruct = Len(mxcd)
        mxcd.dwControlID = mxc.dwControlID
        mxcd.cChannels = 1
        mxcd.item = 0
        mxcd.paDetails = VarPtr(mxcd_u)
        mxcd.cbDetails = Len(mxcd_u)
'
        mxcd_u.dwValue = Volume
'
        lRet = mixerSetControlDetails(hmixer, mxcd, MIXER_SETCONTROLDETAILSF_VALUE)
        If (lRet <> MMSYSERR_NOERROR) Then
            Err.Raise 1001, , "mixerSetControlDetails - Err: " & WaveERR(lRet)
        End If
    End If
'
'
ImpostaVolumeLinea_ERR:
    lRet = mixerClose(hmixer)
'
    If (Err.Number <> 0) Then
        MsgBox Err.Number & vbNewLine & Err.Description, _
               vbCritical, " modMixerControl: ImpostaVolumeLinea"
        Err.Clear
    End If
'
'
'
End Sub
Private Function BytesToString(Char() As Byte) As String
'
'   Converte in stringhe i vettori di bytes ritornati
'   dalle API nelle variabili di tipo CHAR[NCHARS]:
'
    Dim StringT$
'
    StringT$ = StrConv(Char(), vbUnicode)
    BytesToString = Left$(StringT$, InStr(StringT$, Chr$(0)) - 1)
'
'
'
End Function
Public Function WaveERR(ByVal lRet As Long) As String
'
'   Ritorna la descrizione dell' errore lRet eventualmente generato
'   dalle "Waveform Functions" e dalle "Audio Mixer Functions":
'
    Dim WFErr$
'
    Select Case lRet
    Case MMSYSERR_ERROR
        WFErr$ = "unspecified error"
'
    Case MMSYSERR_BADDEVICEID
        WFErr$ = "device ID out of range"
'
    Case MMSYSERR_NOTENABLED
        WFErr$ = "driver failed enable"
'
    Case MMSYSERR_ALLOCATED
        WFErr$ = "device already allocated"
'
    Case MMSYSERR_INVALHANDLE
        WFErr$ = "device handle is invalid"
'
    Case MMSYSERR_NODRIVER
        WFErr$ = "no device driver present"
'
    Case MMSYSERR_NOMEM
        WFErr$ = "memory allocation error"
'
    Case MMSYSERR_NOTSUPPORTED
        WFErr$ = "function isn't supported"
'
    Case MMSYSERR_INVALFLAG
        WFErr$ = "invalid flag passed"
'
    Case MMSYSERR_INVALPARAM
        WFErr$ = "invalid parameter passed"
'
    Case MMSYSERR_HANDLEBUSY
        WFErr$ = "handle being used"
'
    Case MMSYSERR_INVALIDALIAS
        WFErr$ = "specified alias not found in WIN.INI"
'
    Case WAVERR_BADFORMAT
        WFErr$ = "unsupported wave format"
'
    Case WAVERR_STILLPLAYING
        WFErr$ = "still something playing"
'
    Case WAVERR_UNPREPARED
        WFErr$ = "header not prepared"
'
    Case WAVERR_SYNC
        WFErr$ = "device is synchronous"
'
    Case MIXERR_INVALLINE
        WFErr$ = "the audio line reference is invalid"
'
    Case MIXERR_INVALCONTROL
        WFErr$ = "the control reference is invalid"
    
    Case MIXERR_INVALVALUE
        WFErr$ = "The value specified for the mixer control is not within the limits for the control"
'
    Case Else
        WFErr$ = "unspecified error"
    End Select
'
    WaveERR = Str$(lRet) & " - " & WFErr$
'
'
'
End Function

Public Function CercaNomiLineeAttive(Mixer_I As Mixer_Type, _
    Optional ByRef I1 As Long) As String
'
'   Ritorna una stringa contenente i nomi delle "Source lines" attive
'   nel mixer Mixer_I; ritorna anche I1 che e' l' indice della prima
'   linea attiva (se non vi sono linee attive ritorna I1 = -1):
'
    Dim I&, Nomi$
'
    On Error GoTo CercaNomiLineeAttive_ERR
'
    I1 = -1
    For I = 0 To Mixer_I.NSrcLines - 1
        If Mixer_I.SrcLine(I).bNotMute Then
            If (Nomi$ <> "") Then Nomi$ = Nomi$ & ", "
            Nomi$ = Nomi$ & Mixer_I.SrcLine(I).Name
'
            If (I1 < 0) Then I1 = I
        End If
    Next I
'
    CercaNomiLineeAttive = Nomi$
'
'
CercaNomiLineeAttive_ERR:
    If (Err.Number <> 0) Then
        CercaNomiLineeAttive = "??"
        Err.Clear
    End If
'
'
'
End Function
